<?php
/**
 * Plugin Name: Kadeřnice Rezervace
 * Plugin URI: https://www.test.rezervace-atelier.eu/
 * Description: Kompletní rezervační systém pro kadeřnictví s kalendářem, správou služeb a online rezervacemi
 * Version: 1.0.0
 * Author: Váš tým
 * Author URI: https://www.test.rezervace-atelier.eu/
 * Text Domain: kadernice-rezervace
 * Domain Path: /languages
 * Requires at least: 6.0
 * Requires PHP: 7.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('KR_VERSION', '1.0.0');
define('KR_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('KR_PLUGIN_URL', plugin_dir_url(__FILE__));
define('KR_PLUGIN_FILE', __FILE__);

// Include required files
require_once KR_PLUGIN_DIR . 'includes/class-kr-activator.php';
require_once KR_PLUGIN_DIR . 'includes/class-kr-deactivator.php';
require_once KR_PLUGIN_DIR . 'includes/class-kr-database.php';
require_once KR_PLUGIN_DIR . 'includes/class-kr-services.php';
require_once KR_PLUGIN_DIR . 'includes/class-kr-bookings.php';
require_once KR_PLUGIN_DIR . 'includes/class-kr-email.php';
require_once KR_PLUGIN_DIR . 'includes/class-kr-sms.php';
require_once KR_PLUGIN_DIR . 'includes/class-kr-shortcode.php';
require_once KR_PLUGIN_DIR . 'includes/class-kr-gutenberg.php';
require_once KR_PLUGIN_DIR . 'includes/class-kr-ajax.php';
require_once KR_PLUGIN_DIR . 'includes/class-kr-admin.php';
require_once KR_PLUGIN_DIR . 'includes/class-kr-welcome.php';
require_once KR_PLUGIN_DIR . 'includes/class-kr-license.php';
require_once KR_PLUGIN_DIR . 'includes/class-kr-shop.php';
require_once KR_PLUGIN_DIR . 'includes/class-kr-shop-handler.php';
require_once KR_PLUGIN_DIR . 'includes/class-kr-integration.php';

/**
 * Plugin activation hook
 */
function activate_kadernice_rezervace() {
    KR_Activator::activate();
}
register_activation_hook(__FILE__, 'activate_kadernice_rezervace');

/**
 * Plugin deactivation hook
 */
function deactivate_kadernice_rezervace() {
    KR_Deactivator::deactivate();
}
register_deactivation_hook(__FILE__, 'deactivate_kadernice_rezervace');

/**
 * Main plugin class
 */
class Kadernice_Rezervace {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->init();
    }
    
    private function init() {
        // Initialize components
        add_action('plugins_loaded', array($this, 'load_textdomain'));
        add_action('init', array($this, 'init_components'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
    }
    
    public function load_textdomain() {
        load_plugin_textdomain('kadernice-rezervace', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }
    
    public function init_components() {
        // Initialize shortcode
        new KR_Shortcode();
        
        // Initialize Gutenberg block
        new KR_Gutenberg();
        
        // Initialize AJAX handlers
        new KR_Ajax();
        
        // Initialize admin
        if (is_admin()) {
            new KR_Admin();
        }
    }
    
    public function enqueue_frontend_assets() {
        // Enqueue styles
        wp_enqueue_style(
            'kr-frontend-style',
            KR_PLUGIN_URL . 'assets/css/frontend.css',
            array(),
            KR_VERSION
        );
        
        // Enqueue scripts
        wp_enqueue_script(
            'kr-frontend-script',
            KR_PLUGIN_URL . 'assets/js/frontend.js',
            array('jquery'),
            KR_VERSION,
            true
        );
        
        // Localize script
        wp_localize_script('kr-frontend-script', 'krAjax', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('kr-ajax-nonce'),
            'strings' => array(
                'loading' => __('Načítání...', 'kadernice-rezervace'),
                'error' => __('Došlo k chybě. Zkuste to prosím znovu.', 'kadernice-rezervace'),
                'errorWithContact' => get_option('kr_error_message', 
                    __('Něco se pokazilo. Prosím kontaktujte nás na telefonu ', 'kadernice-rezervace') . 
                    get_option('kr_business_phone', '')
                ),
                'success' => __('Rezervace byla úspěšně odeslána!', 'kadernice-rezervace'),
                'selectDate' => __('Vyberte datum', 'kadernice-rezervace'),
                'selectTime' => __('Vyberte čas', 'kadernice-rezervace'),
                'selectService' => __('Vyberte službu', 'kadernice-rezervace'),
            )
        ));
    }
    
    public function enqueue_admin_assets($hook) {
        // Only load on plugin pages
        if (strpos($hook, 'kadernice-rezervace') === false) {
            return;
        }
        
        // Enqueue admin styles
        wp_enqueue_style(
            'kr-admin-style',
            KR_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            KR_VERSION
        );
        
        // Enqueue admin scripts
        wp_enqueue_script(
            'kr-admin-script',
            KR_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery', 'jquery-ui-datepicker'),
            KR_VERSION,
            true
        );
        
        // Localize admin script
        wp_localize_script('kr-admin-script', 'krAdmin', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('kr-admin-nonce')
        ));
    }
}
// Initialize AJAX handler
add_action('plugins_loaded', function() {
    new KR_Ajax();
});

// Initialize the plugin
function kadernice_rezervace() {
    return Kadernice_Rezervace::get_instance();
}

// Start the plugin
kadernice_rezervace();