<?php
/**
 * Welcome screen handler
 */

if (!defined('ABSPATH')) {
    exit;
}

class KR_Welcome {
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_welcome_page'));
        add_action('admin_init', array($this, 'handle_activation_redirect'));
        add_action('admin_init', array($this, 'handle_welcome_actions'));
        add_action('admin_notices', array($this, 'show_welcome_notice'));
    }
    
    /**
     * Add welcome page
     */
    public function add_welcome_page() {
        $hook = add_dashboard_page(
            __('Vítejte v Kadeřnice Rezervace', 'kadernice-rezervace'),
            __('Kadeřnice Rezervace', 'kadernice-rezervace'),
            'manage_options',
            'kr-welcome',
            array($this, 'render_welcome_page')
        );
        
        // Set page title
        add_action('load-' . $hook, array($this, 'set_page_title'));
        
        // Remove from menu
        remove_submenu_page('index.php', 'kr-welcome');
    }
    
    /**
     * Set page title
     */
    public function set_page_title() {
        global $title;
        $title = __('Vítejte v Kadeřnice Rezervace', 'kadernice-rezervace');
    }
    
    /**
     * Handle activation redirect
     */
    public function handle_activation_redirect() {
        // Check if we should redirect
        if (get_transient('kr_activation_redirect')) {
            delete_transient('kr_activation_redirect');
            
            // Don't redirect on multi-activation
            if (isset($_GET['activate-multi'])) {
                return;
            }
            
            // Redirect to main admin page (welcome/dashboard)
            wp_safe_redirect(admin_url('admin.php?page=kadernice-rezervace'));
            exit;
        }
    }
    
    /**
     * Show welcome notice
     */
    public function show_welcome_notice() {
        // Don't show if already dismissed or page created
        if (get_option('kr_welcome_dismissed') || get_option('kr_booking_page_id')) {
            return;
        }
        
        // Don't show on plugin pages
        if (isset($_GET['page']) && strpos($_GET['page'], 'kadernice-rezervace') !== false) {
            return;
        }
        if (isset($_GET['page']) && strpos($_GET['page'], 'kr-') !== false) {
            return;
        }
        
        $welcome_url = admin_url('admin.php?page=kadernice-rezervace');
        ?>
        <div class="notice notice-info is-dismissible kr-welcome-notice">
            <p>
                <strong><?php _e('Kadeřnice Rezervace je aktivní!', 'kadernice-rezervace'); ?></strong>
                <?php _e('Dokončete nastavení a vytvořte rezervační stránku.', 'kadernice-rezervace'); ?>
                <a href="<?php echo esc_url($welcome_url); ?>" class="button button-primary" style="margin-left: 10px;">
                    <?php _e('Přejít na úvodní stránku', 'kadernice-rezervace'); ?>
                </a>
            </p>
        </div>
        <script>
        jQuery(document).ready(function($) {
            $(document).on('click', '.kr-welcome-notice .notice-dismiss', function() {
                $.post(ajaxurl, {
                    action: 'kr_dismiss_welcome',
                    nonce: '<?php echo wp_create_nonce('kr-dismiss-welcome'); ?>'
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * Handle welcome page actions
     */
    public function handle_welcome_actions() {
        // Dismiss welcome notice
        add_action('wp_ajax_kr_dismiss_welcome', function() {
            check_ajax_referer('kr-dismiss-welcome', 'nonce');
            update_option('kr_welcome_dismissed', 1);
            wp_send_json_success();
        });
        
        // Create booking page
        if (isset($_POST['kr_create_page']) && check_admin_referer('kr_create_page_nonce')) {
            $page_id = $this->create_booking_page();
            
            if ($page_id) {
                update_option('kr_booking_page_id', $page_id);
                wp_safe_redirect(admin_url('post.php?post=' . $page_id . '&action=edit'));
                exit;
            }
        }
    }
    
    /**
     * Create booking page
     */
    private function create_booking_page() {
        $page_data = array(
            'post_title' => __('Rezervace', 'kadernice-rezervace'),
            'post_content' => '[kadernice_rezervace]',
            'post_status' => 'publish',
            'post_type' => 'page',
            'post_author' => get_current_user_id(),
            'comment_status' => 'closed',
            'ping_status' => 'closed'
        );
        
        $page_id = wp_insert_post($page_data);
        
        return $page_id;
    }
    
    /**
     * Render welcome page
     */
    public function render_welcome_page() {
        $booking_page_id = get_option('kr_booking_page_id');
        $booking_page = $booking_page_id ? get_post($booking_page_id) : null;
        ?>
        <div class="wrap kr-welcome-wrap">
            <style>
                .kr-welcome-wrap {
                    max-width: 1200px;
                    margin: 40px auto;
                }
                .kr-welcome-header {
                    background: linear-gradient(135deg, #f89820 0%, #e08718 100%);
                    color: white;
                    padding: 60px 40px;
                    border-radius: 8px;
                    text-align: center;
                    margin-bottom: 40px;
                    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
                }
                .kr-welcome-header h1 {
                    color: white;
                    font-size: 42px;
                    margin: 0 0 16px;
                    font-weight: 700;
                }
                .kr-welcome-header p {
                    font-size: 18px;
                    margin: 0;
                    opacity: 0.95;
                }
                .kr-welcome-grid {
                    display: grid;
                    grid-template-columns: 1fr 1fr;
                    gap: 30px;
                    margin-bottom: 40px;
                }
                .kr-welcome-card {
                    background: white;
                    border: 1px solid #ddd;
                    border-radius: 8px;
                    padding: 30px;
                    box-shadow: 0 2px 8px rgba(0,0,0,0.05);
                }
                .kr-welcome-card h2 {
                    margin-top: 0;
                    font-size: 24px;
                    color: #2d4a45;
                    display: flex;
                    align-items: center;
                    gap: 12px;
                }
                .kr-welcome-card h2 .dashicons {
                    color: #f89820;
                    font-size: 32px;
                    width: 32px;
                    height: 32px;
                }
                .kr-shortcode-box {
                    background: #f9fafb;
                    border: 2px dashed #e5e7eb;
                    border-radius: 6px;
                    padding: 20px;
                    margin: 20px 0;
                    position: relative;
                }
                .kr-shortcode-code {
                    font-family: 'Courier New', monospace;
                    font-size: 18px;
                    color: #2d4a45;
                    font-weight: 600;
                    text-align: center;
                    user-select: all;
                }
                .kr-copy-button {
                    position: absolute;
                    top: 10px;
                    right: 10px;
                    background: #f89820;
                    color: white;
                    border: none;
                    padding: 8px 16px;
                    border-radius: 4px;
                    cursor: pointer;
                    font-size: 13px;
                    font-weight: 600;
                    transition: all 0.3s;
                }
                .kr-copy-button:hover {
                    background: #e08718;
                }
                .kr-copy-button.copied {
                    background: #22c55e;
                }
                .kr-steps-list {
                    list-style: none;
                    padding: 0;
                    margin: 20px 0;
                }
                .kr-steps-list li {
                    padding: 12px 0 12px 40px;
                    position: relative;
                    line-height: 1.6;
                    color: #4b5563;
                }
                .kr-steps-list li:before {
                    content: "✓";
                    position: absolute;
                    left: 0;
                    top: 12px;
                    width: 24px;
                    height: 24px;
                    background: #22c55e;
                    color: white;
                    border-radius: 50%;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    font-weight: bold;
                    font-size: 14px;
                }
                .kr-action-buttons {
                    display: flex;
                    gap: 12px;
                    margin-top: 24px;
                }
                .kr-btn-primary {
                    background: #f89820;
                    color: white;
                    border: none;
                    padding: 14px 28px;
                    border-radius: 6px;
                    font-size: 16px;
                    font-weight: 600;
                    cursor: pointer;
                    text-decoration: none;
                    display: inline-block;
                    transition: all 0.3s;
                }
                .kr-btn-primary:hover {
                    background: #e08718;
                    color: white;
                    transform: translateY(-2px);
                    box-shadow: 0 4px 12px rgba(248, 152, 32, 0.3);
                }
                .kr-btn-secondary {
                    background: white;
                    color: #2d4a45;
                    border: 2px solid #2d4a45;
                    padding: 12px 26px;
                    border-radius: 6px;
                    font-size: 16px;
                    font-weight: 600;
                    cursor: pointer;
                    text-decoration: none;
                    display: inline-block;
                    transition: all 0.3s;
                }
                .kr-btn-secondary:hover {
                    background: #2d4a45;
                    color: white;
                }
                .kr-success-message {
                    background: #d1fae5;
                    border: 1px solid #22c55e;
                    border-radius: 6px;
                    padding: 16px;
                    margin: 20px 0;
                    color: #065f46;
                }
                .kr-success-message .dashicons {
                    color: #22c55e;
                    margin-right: 8px;
                }
                .kr-video-placeholder {
                    background: #f3f4f6;
                    border-radius: 8px;
                    padding: 60px 20px;
                    text-align: center;
                    color: #6b7280;
                    margin: 20px 0;
                }
                @media (max-width: 782px) {
                    .kr-welcome-grid {
                        grid-template-columns: 1fr;
                    }
                    .kr-welcome-header {
                        padding: 40px 20px;
                    }
                    .kr-welcome-header h1 {
                        font-size: 32px;
                    }
                }
            </style>
            
            <div class="kr-welcome-header">
                <h1><?php _e('Vítejte v Kadeřnice Rezervace! 💇‍♀️', 'kadernice-rezervace'); ?></h1>
                <p><?php _e('Váš nový rezervační systém je připraven. Pojďme ho nastavit!', 'kadernice-rezervace'); ?></p>
            </div>
            
            <div class="kr-welcome-grid">
                <!-- Quick Start -->
                <div class="kr-welcome-card">
                    <h2>
                        <span class="dashicons dashicons-calendar-alt"></span>
                        <?php _e('Rychlé vytvoření stránky', 'kadernice-rezervace'); ?>
                    </h2>
                    
                    <?php if ($booking_page): ?>
                        <div class="kr-success-message">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <strong><?php _e('Stránka vytvořena!', 'kadernice-rezervace'); ?></strong>
                            <p style="margin: 8px 0 0;">
                                <?php printf(
                                    __('Rezervační stránka "%s" byla úspěšně vytvořena.', 'kadernice-rezervace'),
                                    esc_html($booking_page->post_title)
                                ); ?>
                            </p>
                        </div>
                        
                        <div class="kr-action-buttons">
                            <a href="<?php echo get_permalink($booking_page_id); ?>" class="kr-btn-primary" target="_blank">
                                <?php _e('Zobrazit stránku', 'kadernice-rezervace'); ?>
                            </a>
                            <a href="<?php echo admin_url('post.php?post=' . $booking_page_id . '&action=edit'); ?>" class="kr-btn-secondary">
                                <?php _e('Upravit stránku', 'kadernice-rezervace'); ?>
                            </a>
                        </div>
                    <?php else: ?>
                        <p><?php _e('Kliknutím na tlačítko níže automaticky vytvoříte novou stránku s rezervačním formulářem.', 'kadernice-rezervace'); ?></p>
                        
                        <form method="post" action="">
                            <?php wp_nonce_field('kr_create_page_nonce'); ?>
                            <div class="kr-action-buttons">
                                <button type="submit" name="kr_create_page" class="kr-btn-primary">
                                    <?php _e('Vytvořit rezervační stránku', 'kadernice-rezervace'); ?>
                                </button>
                            </div>
                        </form>
                        
                        <p style="margin-top: 20px; color: #6b7280; font-size: 14px;">
                            <?php _e('Stránka bude vytvořena s názvem "Rezervace" a bude obsahovat shortcode [kadernice_rezervace]', 'kadernice-rezervace'); ?>
                        </p>
                    <?php endif; ?>
                </div>
                
                <!-- Manual Setup -->
                <div class="kr-welcome-card">
                    <h2>
                        <span class="dashicons dashicons-editor-code"></span>
                        <?php _e('Manuální vložení', 'kadernice-rezervace'); ?>
                    </h2>
                    
                    <p><?php _e('Zkopírujte tento shortcode a vložte ho na libovolnou stránku nebo do příspěvku:', 'kadernice-rezervace'); ?></p>
                    
                    <div class="kr-shortcode-box">
                        <button type="button" class="kr-copy-button" onclick="krCopyShortcode(this)">
                            <?php _e('Zkopírovat', 'kadernice-rezervace'); ?>
                        </button>
                        <div class="kr-shortcode-code">[kadernice_rezervace]</div>
                    </div>
                    
                    <p style="margin-top: 20px;">
                        <strong><?php _e('Nebo použijte Gutenberg blok:', 'kadernice-rezervace'); ?></strong>
                    </p>
                    <ol style="margin-left: 20px; color: #6b7280;">
                        <li><?php _e('Vytvořte novou stránku', 'kadernice-rezervace'); ?></li>
                        <li><?php _e('Klikněte na "+" pro přidání bloku', 'kadernice-rezervace'); ?></li>
                        <li><?php _e('Vyhledejte "Kadeřnice Rezervace"', 'kadernice-rezervace'); ?></li>
                        <li><?php _e('Přidejte blok a publikujte', 'kadernice-rezervace'); ?></li>
                    </ol>
                </div>
            </div>
            
            <!-- Setup Steps -->
            <div class="kr-welcome-card" style="margin-bottom: 40px;">
                <h2>
                    <span class="dashicons dashicons-admin-settings"></span>
                    <?php _e('Doporučené kroky nastavení', 'kadernice-rezervace'); ?>
                </h2>
                
                <ul class="kr-steps-list">
                    <li>
                        <strong><?php _e('Základní informace', 'kadernice-rezervace'); ?></strong><br>
                        <?php _e('Vyplňte název provozovny, email a telefon v', 'kadernice-rezervace'); ?>
                        <a href="<?php echo admin_url('admin.php?page=kr-settings'); ?>"><?php _e('Nastavení', 'kadernice-rezervace'); ?></a>
                    </li>
                    <li>
                        <strong><?php _e('Upravte služby', 'kadernice-rezervace'); ?></strong><br>
                        <?php _e('Přizpůsobte služby, ceny a dobu trvání v sekci', 'kadernice-rezervace'); ?>
                        <a href="<?php echo admin_url('admin.php?page=kr-services'); ?>"><?php _e('Služby', 'kadernice-rezervace'); ?></a>
                    </li>
                    <li>
                        <strong><?php _e('Nastavte pracovní dobu', 'kadernice-rezervace'); ?></strong><br>
                        <?php _e('Určete, kdy jste otevřeni v sekci', 'kadernice-rezervace'); ?>
                        <a href="<?php echo admin_url('admin.php?page=kr-working-hours'); ?>"><?php _e('Pracovní doba', 'kadernice-rezervace'); ?></a>
                    </li>
                    <li>
                        <strong><?php _e('Otestujte rezervaci', 'kadernice-rezervace'); ?></strong><br>
                        <?php _e('Proveďte testovací rezervaci a zkontrolujte, že vše funguje správně', 'kadernice-rezervace'); ?>
                    </li>
                </ul>
                
                <div class="kr-action-buttons">
                    <a href="<?php echo admin_url('admin.php?page=kr-settings'); ?>" class="kr-btn-primary">
                        <?php _e('Přejít do nastavení', 'kadernice-rezervace'); ?>
                    </a>
                    <a href="<?php echo admin_url('admin.php?page=kadernice-rezervace'); ?>" class="kr-btn-secondary">
                        <?php _e('Zobrazit rezervace', 'kadernice-rezervace'); ?>
                    </a>
                </div>
            </div>
            
            <!-- Help -->
            <div class="kr-welcome-card">
                <h2>
                    <span class="dashicons dashicons-sos"></span>
                    <?php _e('Potřebujete pomoc?', 'kadernice-rezervace'); ?>
                </h2>
                
                <p><?php _e('Máte dotazy nebo potřebujete podporu? Jsme tu pro vás!', 'kadernice-rezervace'); ?></p>
                
                <ul style="margin-left: 20px; color: #6b7280;">
                    <li>
                        <strong><?php _e('Dokumentace:', 'kadernice-rezervace'); ?></strong>
                        <a href="<?php echo KR_PLUGIN_DIR . 'README.md'; ?>" target="_blank">README.md</a>
                    </li>
                    <li>
                        <strong><?php _e('Email podpora:', 'kadernice-rezervace'); ?></strong>
                        podpora@example.com
                    </li>
                    <li>
                        <strong><?php _e('Časté dotazy:', 'kadernice-rezervace'); ?></strong>
                        <a href="https://example.com/faq" target="_blank"><?php _e('FAQ sekce', 'kadernice-rezervace'); ?></a>
                    </li>
                </ul>
            </div>
            
            <script>
            function krCopyShortcode(button) {
                const shortcode = '[kadernice_rezervace]';
                
                // Copy to clipboard
                navigator.clipboard.writeText(shortcode).then(function() {
                    // Change button text
                    const originalText = button.textContent;
                    button.textContent = '<?php esc_js(_e('Zkopírováno!', 'kadernice-rezervace')); ?>';
                    button.classList.add('copied');
                    
                    // Reset after 2 seconds
                    setTimeout(function() {
                        button.textContent = originalText;
                        button.classList.remove('copied');
                    }, 2000);
                }).catch(function(err) {
                    alert('<?php esc_js(_e('Chyba při kopírování. Zkuste to prosím ručně.', 'kadernice-rezervace')); ?>');
                });
            }
            </script>
        </div>
        <?php
    }
}

// Initialize welcome screen
new KR_Welcome();