<?php
/**
 * Plugin uninstallation handler
 * Handles cleanup when plugin is deleted (not just deactivated)
 */

if (!defined('ABSPATH')) {
    exit;
}

class KR_Uninstaller {
    
    /**
     * Run uninstall procedures
     * This should be called from uninstall.php
     */
    public static function uninstall() {
        global $wpdb;
        
        // Security check - only run if user has permission
        if (!current_user_can('activate_plugins')) {
            return;
        }
        
        // Delete all plugin tables
        self::delete_tables();
        
        // Delete all plugin options
        self::delete_options();
        
        // Clear any scheduled events
        self::clear_scheduled_events();
        
        // Clear transients
        self::clear_transients();
    }
    
    /**
     * Delete all plugin database tables
     */
    private static function delete_tables() {
        global $wpdb;
        
        $tables = array(
            $wpdb->prefix . 'kr_services',
            $wpdb->prefix . 'kr_bookings',
            $wpdb->prefix . 'kr_time_slots',
            $wpdb->prefix . 'kr_blocked_dates'
        );
        
        foreach ($tables as $table) {
            $wpdb->query("DROP TABLE IF EXISTS $table");
        }
    }
    
    /**
     * Delete all plugin options
     */
    private static function delete_options() {
        $options = array(
            'kr_business_name',
            'kr_business_email',
            'kr_business_phone',
            'kr_booking_interval',
            'kr_advance_booking_days',
            'kr_min_booking_notice',
            'kr_max_bookings_per_slot',
            'kr_enable_email_notifications',
            'kr_admin_email_on_booking',
            'kr_customer_email_on_booking',
            'kr_currency',
            'kr_date_format',
            'kr_time_format',
            'kr_error_message',
            'kr_language',
            'kr_payment_gateway_enabled',
            'kr_payment_gateway_type',
            'kr_payment_gateway_api_key',
            'kr_payment_gateway_secret',
            'kr_enable_tips',
            'kr_activation_redirect'
        );
        
        foreach ($options as $option) {
            delete_option($option);
        }
    }
    
    /**
     * Clear scheduled events
     */
    private static function clear_scheduled_events() {
        wp_clear_scheduled_hook('kr_daily_cleanup');
        wp_clear_scheduled_hook('kr_send_reminders');
    }
    
    /**
     * Clear plugin transients
     */
    private static function clear_transients() {
        global $wpdb;
        
        $wpdb->query(
            "DELETE FROM {$wpdb->options} 
             WHERE option_name LIKE '_transient_kr_%' 
             OR option_name LIKE '_transient_timeout_kr_%'"
        );
    }
}