<?php
/**
 * SMS notifications handler
 */

if (!defined('ABSPATH')) {
    exit;
}

class KR_SMS {
    
    /**
     * Send booking confirmation SMS to customer
     */
    public static function send_booking_confirmation($booking_id) {
        if (!get_option('kr_sms_enabled') || !get_option('kr_sms_customer_notify')) {
            return;
        }
        
        $booking = KR_Database::get_booking($booking_id);
        if (!$booking) {
            return;
        }
        
        $message = self::get_booking_confirmation_message($booking);
        return self::send_sms($booking->customer_phone, $message);
    }
    
    /**
     * Send notification to admin
     */
    public static function send_admin_notification($booking_id) {
        if (!get_option('kr_sms_enabled') || !get_option('kr_sms_admin_notify')) {
            return;
        }
        
        $booking = KR_Database::get_booking($booking_id);
        if (!$booking) {
            return;
        }
        
        $admin_phone = get_option('kr_business_phone');
        if (empty($admin_phone)) {
            return;
        }
        
        $message = self::get_admin_notification_message($booking);
        return self::send_sms($admin_phone, $message);
    }
    
    /**
     * Send reminder SMS
     */
    public static function send_reminder($booking_id) {
        if (!get_option('kr_sms_enabled') || !get_option('kr_sms_reminder_enabled')) {
            return;
        }
        
        $booking = KR_Database::get_booking($booking_id);
        if (!$booking) {
            return;
        }
        
        $message = self::get_reminder_message($booking);
        return self::send_sms($booking->customer_phone, $message);
    }
    
    /**
     * Send SMS via selected provider
     */
    private static function send_sms($to, $message) {
        $provider = get_option('kr_sms_provider', 'twilio');
        
        switch ($provider) {
            case 'twilio':
                return self::send_via_twilio($to, $message);
            case 'smsbrana':
                return self::send_via_smsbrana($to, $message);
            case 'nexmo':
                return self::send_via_nexmo($to, $message);
            default:
                return array('success' => false, 'message' => 'Unknown provider');
        }
    }
    
    /**
     * Send via Twilio
     */
    private static function send_via_twilio($to, $message) {
        $api_key = get_option('kr_sms_api_key');
        $api_secret = get_option('kr_sms_api_secret');
        $sender_id = get_option('kr_sms_sender_id');
        
        if (empty($api_key) || empty($api_secret)) {
            return array('success' => false, 'message' => 'Missing API credentials');
        }
        
        $url = "https://api.twilio.com/2010-04-01/Accounts/{$api_key}/Messages.json";
        
        $data = array(
            'From' => $sender_id,
            'To' => $to,
            'Body' => $message
        );
        
        $response = wp_remote_post($url, array(
            'headers' => array(
                'Authorization' => 'Basic ' . base64_encode($api_key . ':' . $api_secret)
            ),
            'body' => $data
        ));
        
        if (is_wp_error($response)) {
            return array('success' => false, 'message' => $response->get_error_message());
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($body['sid'])) {
            return array('success' => true, 'message' => 'SMS sent successfully');
        }
        
        return array('success' => false, 'message' => $body['message'] ?? 'Unknown error');
    }
    
    /**
     * Send via SMSBrana.cz
     */
    private static function send_via_smsbrana($to, $message) {
        $login = get_option('kr_sms_api_key');
        $password = get_option('kr_sms_api_secret');
        
        if (empty($login) || empty($password)) {
            return array('success' => false, 'message' => 'Missing API credentials');
        }
        
        $url = 'https://api.smsbrana.cz/smsconnect/http.php';
        
        $data = array(
            'action' => 'send_sms',
            'login' => $login,
            'password' => $password,
            'number' => $to,
            'message' => $message
        );
        
        $response = wp_remote_post($url, array(
            'body' => $data
        ));
        
        if (is_wp_error($response)) {
            return array('success' => false, 'message' => $response->get_error_message());
        }
        
        $body = wp_remote_retrieve_body($response);
        
        if (strpos($body, 'OK') === 0) {
            return array('success' => true, 'message' => 'SMS sent successfully');
        }
        
        return array('success' => false, 'message' => $body);
    }
    
    /**
     * Send via Nexmo (Vonage)
     */
    private static function send_via_nexmo($to, $message) {
        $api_key = get_option('kr_sms_api_key');
        $api_secret = get_option('kr_sms_api_secret');
        $sender_id = get_option('kr_sms_sender_id', 'KADERNICE');
        
        if (empty($api_key) || empty($api_secret)) {
            return array('success' => false, 'message' => 'Missing API credentials');
        }
        
        $url = 'https://rest.nexmo.com/sms/json';
        
        $data = array(
            'api_key' => $api_key,
            'api_secret' => $api_secret,
            'from' => $sender_id,
            'to' => $to,
            'text' => $message
        );
        
        $response = wp_remote_post($url, array(
            'body' => json_encode($data),
            'headers' => array('Content-Type' => 'application/json')
        ));
        
        if (is_wp_error($response)) {
            return array('success' => false, 'message' => $response->get_error_message());
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($body['messages'][0]['status']) && $body['messages'][0]['status'] == '0') {
            return array('success' => true, 'message' => 'SMS sent successfully');
        }
        
        return array('success' => false, 'message' => $body['messages'][0]['error-text'] ?? 'Unknown error');
    }
    
    /**
     * Get booking confirmation message
     */
    private static function get_booking_confirmation_message($booking) {
        $date = date_i18n(get_option('kr_date_format', 'd.m.Y'), strtotime($booking->booking_date));
        $time = date_i18n(get_option('kr_time_format', 'H:i'), strtotime($booking->booking_time));
        
        $message = sprintf(
            __('Potvrzeni rezervace: %s, %s v %s. Dekujeme!', 'kadernice-rezervace'),
            $booking->service_name,
            $date,
            $time
        );
        
        return $message;
    }
    
    /**
     * Get admin notification message
     */
    private static function get_admin_notification_message($booking) {
        $date = date_i18n(get_option('kr_date_format', 'd.m.Y'), strtotime($booking->booking_date));
        $time = date_i18n(get_option('kr_time_format', 'H:i'), strtotime($booking->booking_time));
        
        $message = sprintf(
            __('Nova rezervace: %s - %s, %s v %s', 'kadernice-rezervace'),
            $booking->customer_name,
            $booking->service_name,
            $date,
            $time
        );
        
        return $message;
    }
    
    /**
     * Get reminder message
     */
    private static function get_reminder_message($booking) {
        $date = date_i18n(get_option('kr_date_format', 'd.m.Y'), strtotime($booking->booking_date));
        $time = date_i18n(get_option('kr_time_format', 'H:i'), strtotime($booking->booking_time));
        $hours = get_option('kr_sms_reminder_hours', 24);
        
        $message = sprintf(
            __('Pripominani: Za %d hodin mate rezervaci %s v %s. Tesime se na Vas!', 'kadernice-rezervace'),
            $hours,
            $date,
            $time
        );
        
        return $message;
    }
    
    /**
     * Send test SMS
     */
    public static function send_test_sms($phone) {
        $message = __('Testovaci SMS z pluginu Kadernice Rezervace. Vse funguje spravne!', 'kadernice-rezervace');
        return self::send_sms($phone, $message);
    }
}