<?php
/**
 * Shop payment handler for Kadeřnice Rezervace
 */

if (!defined('ABSPATH')) {
    exit;
}

class KR_Shop_Handler {
    
    public function __construct() {
        error_log('KR Shop Handler: Class initialized');
        add_action('wp_ajax_kr_process_shop_payment', array($this, 'process_payment'));
        add_action('wp_ajax_nopriv_kr_process_shop_payment', array($this, 'process_payment'));
        error_log('KR Shop Handler: AJAX actions registered');
    }
    
    /**
     * Process shop payment
     */
    public function process_payment() {
        error_log('KR Shop Handler: process_payment() called');
        error_log('KR Shop Handler: POST data: ' . print_r($_POST, true));
        
        // Verify nonce
        check_ajax_referer('kr-shop-nonce', 'nonce');
        
        // Sanitize input
        $name = sanitize_text_field($_POST['name']);
        $email = sanitize_email($_POST['email']);
        $phone = sanitize_text_field($_POST['phone']);
        $product_name = sanitize_text_field($_POST['product_name']);
        $price = floatval($_POST['price']);
        
        // Validate
        if (empty($name) || empty($email) || !is_email($email)) {
            wp_send_json_error(array('message' => __('Vyplňte prosím všechny povinné údaje.', 'kadernice-rezervace')));
            return;
        }
        
        // Load Rezervace Admin functions if not already loaded
        $this->load_rezervace_admin_functions();
        
        // Debug: Log attempt
        error_log('KR Shop: Attempting to create license for ' . $email);
        
        // Check if license already exists
        $existing = $this->get_license_by_email($email, $product_name);
        if ($existing) {
            error_log('KR Shop: License already exists for ' . $email);
            wp_send_json_error(array('message' => __('Pro tento email již existuje licence. Zkontrolujte svou emailovou schránku.', 'kadernice-rezervace')));
            return;
        }
        
        // Create license record in Rezervace Admin database
        $license_data = array(
            'name' => $name,
            'email' => $email,
            'phone' => $phone,
            'product_name' => $product_name,
            'price' => $price,
            'status' => 'čeká na platbu'
        );
        
        error_log('KR Shop: Inserting license with data: ' . print_r($license_data, true));
        
        $license_id = $this->insert_license($license_data);
        
        error_log('KR Shop: License ID returned: ' . ($license_id ? $license_id : 'FALSE'));
        
        if (!$license_id) {
            global $wpdb;
            error_log('KR Shop: Database error: ' . $wpdb->last_error);
            wp_send_json_error(array('message' => __('Chyba při vytváření objednávky. Zkuste to prosím znovu. Error: ' . $wpdb->last_error, 'kadernice-rezervace')));
            return;
        }
        
        // Send to admin email
        $this->send_admin_notification($name, $email, $phone, $product_name, $price, $license_id);
        
        // Send payment instructions to customer
        $this->send_payment_instructions($email, $name, $price, $product_name);
        
        wp_send_json_success(array(
            'message' => __('Objednávka byla vytvořena! Platební instrukce byly odeslány na váš email.', 'kadernice-rezervace')
        ));
    }
    
    /**
     * Send admin notification
     */
    private function send_admin_notification($name, $email, $phone, $product_name, $price, $license_id) {
        $admin_email = get_option('admin_email');
        $subject = "Nová objednávka - $product_name";
        
        $admin_url = admin_url('admin.php?page=rezervace-admin');
        
        $message = "
        <html>
        <body style='font-family: Arial, sans-serif;'>
            <div style='background: #f89820; color: white; padding: 20px; text-align: center;'>
                <h2 style='margin: 0;'>🎉 Nová objednávka pluginu!</h2>
            </div>
            
            <div style='padding: 20px;'>
                <table style='border-collapse: collapse; width: 100%; margin-bottom: 20px;'>
                    <tr>
                        <td style='padding: 10px; border: 1px solid #ddd; background: #f9f9f9;'><strong>ID Licence:</strong></td>
                        <td style='padding: 10px; border: 1px solid #ddd;'>#$license_id</td>
                    </tr>
                    <tr>
                        <td style='padding: 10px; border: 1px solid #ddd; background: #f9f9f9;'><strong>Produkt:</strong></td>
                        <td style='padding: 10px; border: 1px solid #ddd;'>$product_name</td>
                    </tr>
                    <tr>
                        <td style='padding: 10px; border: 1px solid #ddd; background: #f9f9f9;'><strong>Cena:</strong></td>
                        <td style='padding: 10px; border: 1px solid #ddd;'><strong style='color: #f89820; font-size: 18px;'>" . number_format($price, 0, ',', ' ') . " Kč</strong></td>
                    </tr>
                    <tr>
                        <td style='padding: 10px; border: 1px solid #ddd; background: #f9f9f9;'><strong>Jméno:</strong></td>
                        <td style='padding: 10px; border: 1px solid #ddd;'>$name</td>
                    </tr>
                    <tr>
                        <td style='padding: 10px; border: 1px solid #ddd; background: #f9f9f9;'><strong>Email:</strong></td>
                        <td style='padding: 10px; border: 1px solid #ddd;'><a href='mailto:$email'>$email</a></td>
                    </tr>
                    <tr>
                        <td style='padding: 10px; border: 1px solid #ddd; background: #f9f9f9;'><strong>Telefon:</strong></td>
                        <td style='padding: 10px; border: 1px solid #ddd;'>$phone</td>
                    </tr>
                    <tr>
                        <td style='padding: 10px; border: 1px solid #ddd; background: #f9f9f9;'><strong>Status:</strong></td>
                        <td style='padding: 10px; border: 1px solid #ddd;'><span style='background: #ffc107; color: #856404; padding: 5px 10px; border-radius: 4px;'>⏳ Čeká na platbu</span></td>
                    </tr>
                </table>
                
                <div style='text-align: center; margin: 30px 0;'>
                    <a href='$admin_url' style='display: inline-block; background: #f89820; color: white; padding: 15px 30px; text-decoration: none; border-radius: 8px; font-weight: bold;'>
                        📋 Zobrazit v administraci
                    </a>
                </div>
                
                <div style='background: #f0f7ff; padding: 15px; border-left: 4px solid #2271b1; border-radius: 4px;'>
                    <p style='margin: 0;'><strong>💡 Co dál?</strong></p>
                    <ol style='margin: 10px 0 0 0; padding-left: 20px;'>
                        <li>Počkejte na platbu od zákazníka</li>
                        <li>Po přijetí platby změňte status na \"Zaplaceno\"</li>
                        <li>Vygenerujte a odešlete licenční klíč</li>
                    </ol>
                </div>
            </div>
        </body>
        </html>
        ";
        
        $headers = array('Content-Type: text/html; charset=UTF-8');
        
        wp_mail($admin_email, $subject, $message, $headers);
    }
    
    /**
     * Send payment instructions
     */
    private function send_payment_instructions($email, $name, $price, $product_name) {
        $subject = "Platební instrukce - $product_name";
        
        // Get bank details from options (you can add these to settings)
        $bank_account = get_option('kr_bank_account', '123456789/0100');
        $variable_symbol = get_option('kr_variable_symbol', time());
        
        $message = "
        <html>
        <body style='font-family: Arial, sans-serif; line-height: 1.6;'>
            <div style='max-width: 600px; margin: 0 auto; padding: 20px;'>
                <div style='background: #f89820; color: white; padding: 30px; text-align: center; border-radius: 12px 12px 0 0;'>
                    <h1 style='margin: 0;'>Děkujeme za objednávku!</h1>
                </div>
                
                <div style='background: #f9f9f9; padding: 30px; border-radius: 0 0 12px 12px;'>
                    <p>Dobrý den <strong>$name</strong>,</p>
                    
                    <p>Vaše objednávka produktu <strong>$product_name</strong> byla úspěšně vytvořena.</p>
                    
                    <div style='background: white; border: 3px solid #f89820; padding: 25px; margin: 25px 0; border-radius: 12px;'>
                        <h3 style='margin-top: 0; color: #2d4a45;'>💳 Platební údaje:</h3>
                        <table style='width: 100%;'>
                            <tr>
                                <td style='padding: 10px 0;'><strong>Částka:</strong></td>
                                <td style='padding: 10px 0; text-align: right;'><strong style='font-size: 24px; color: #f89820;'>" . number_format($price, 0, ',', ' ') . " Kč</strong></td>
                            </tr>
                            <tr>
                                <td style='padding: 10px 0;'><strong>Číslo účtu:</strong></td>
                                <td style='padding: 10px 0; text-align: right;'><code style='background: #f0f0f0; padding: 5px 10px; border-radius: 4px;'>$bank_account</code></td>
                            </tr>
                            <tr>
                                <td style='padding: 10px 0;'><strong>Variabilní symbol:</strong></td>
                                <td style='padding: 10px 0; text-align: right;'><code style='background: #f0f0f0; padding: 5px 10px; border-radius: 4px;'>$variable_symbol</code></td>
                            </tr>
                            <tr>
                                <td style='padding: 10px 0;'><strong>Zpráva pro příjemce:</strong></td>
                                <td style='padding: 10px 0; text-align: right;'>$email</td>
                            </tr>
                        </table>
                    </div>
                    
                    <h3 style='color: #2d4a45;'>📋 Co dál?</h3>
                    <ol style='line-height: 1.8;'>
                        <li>Proveďte platbu na výše uvedený účet</li>
                        <li>Platbu zpracujeme do 24 hodin</li>
                        <li>Licenční klíč obdržíte na tento email</li>
                        <li>Plugin si stáhnete z WordPress repozitáře</li>
                    </ol>
                    
                    <div style='background: #fff3cd; border-left: 4px solid #ffc107; padding: 15px; margin: 20px 0; border-radius: 4px;'>
                        <p style='margin: 0;'><strong>⚠️ Důležité:</strong> Nezapomeňte uvést správný variabilní symbol a zprávu pro příjemce!</p>
                    </div>
                    
                    <div style='background: #d4edda; border-left: 4px solid #28a745; padding: 15px; margin: 20px 0; border-radius: 4px;'>
                        <h4 style='margin: 0 0 10px 0; color: #155724;'>✅ Co získáte:</h4>
                        <ul style='margin: 0; padding-left: 20px; color: #155724;'>
                            <li>Plnou verzi pluginu bez omezení</li>
                            <li>Automatické aktualizace po dobu 1 roku</li>
                            <li>Technickou podporu</li>
                            <li>Přístup k dokumentaci</li>
                        </ul>
                    </div>
                    
                    <div style='text-align: center; color: #666; font-size: 13px; margin-top: 30px; padding-top: 20px; border-top: 1px solid #ddd;'>
                        <p><strong>Potřebujete pomoc?</strong><br>
                        Kontaktujte nás na: " . get_option('admin_email') . "</p>
                        
                        <p>© " . date('Y') . " Kadeřnice Rezervace</p>
                    </div>
                </div>
            </div>
        </body>
        </html>
        ";
        
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: Kadeřnice Rezervace <' . get_option('admin_email') . '>'
        );
        
        return wp_mail($email, $subject, $message, $headers);
    }
    
    /**
     * Load Rezervace Admin functions
     */
    private function load_rezervace_admin_functions() {
        // Check if Rezervace Admin plugin exists
        $ra_plugin_file = WP_PLUGIN_DIR . '/rezervace-admin/rezervace-admin.php';
        $ra_database_file = WP_PLUGIN_DIR . '/rezervace-admin/includes/database.php';
        
        if (file_exists($ra_database_file)) {
            require_once $ra_database_file;
        }
    }
    
    /**
     * Get license by email (wrapper for ra_get_license_by_email)
     */
    private function get_license_by_email($email, $product_name) {
        if (function_exists('ra_get_license_by_email')) {
            return ra_get_license_by_email($email, $product_name);
        }
        
        // Fallback: Direct database query
        global $wpdb;
        $table_name = $wpdb->prefix . 'ra_licenses';
        
        return $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM $table_name WHERE email = %s AND product_name = %s",
                $email,
                $product_name
            )
        );
    }
    
    /**
     * Insert license (wrapper for ra_insert_license)
     */
    private function insert_license($data) {
        global $wpdb;
        
        // Always use direct database insert for reliability
        $table_name = $wpdb->prefix . 'ra_licenses';
        
        // Check if table exists
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '$table_name'") == $table_name;
        
        if (!$table_exists) {
            error_log('KR Shop: Table ' . $table_name . ' does not exist!');
            return false;
        }
        
        error_log('KR Shop: Table exists, proceeding with insert');
        
        $defaults = array(
            'name' => '',
            'email' => '',
            'phone' => '',
            'status' => 'čeká na platbu',
            'product_name' => 'Kadeřnice Rezervace - Plugin',
            'license_key' => null,
            'price' => 0,
            'payment_id' => null,
            'created_at' => current_time('mysql'),
            'paid_at' => null,
            'expires_at' => null
        );
        
        $data = wp_parse_args($data, $defaults);
        
        error_log('KR Shop: Final data for insert: ' . print_r($data, true));
        
        $result = $wpdb->insert(
            $table_name,
            $data,
            array(
                '%s', // name
                '%s', // email
                '%s', // phone
                '%s', // status
                '%s', // product_name
                '%s', // license_key
                '%f', // price
                '%s', // payment_id
                '%s', // created_at
                '%s', // paid_at
                '%s'  // expires_at
            )
        );
        
        if ($result === false) {
            error_log('KR Shop: Insert failed! Error: ' . $wpdb->last_error);
            error_log('KR Shop: Last query: ' . $wpdb->last_query);
            return false;
        }
        
        $insert_id = $wpdb->insert_id;
        error_log('KR Shop: Insert successful! ID: ' . $insert_id);
        
        return $insert_id;
    }
}

// Initialize
new KR_Shop_Handler();