<?php
/**
 * License management system
 */

if (!defined('ABSPATH')) exit;

class KR_License {
    
    private $license_api_url;
    
    public function __construct() {
        // Pokud je aktivní plugin rezervace-admin, použij URL z jeho nastavení
        if (is_plugin_active('rezervace-admin/rezervace-admin.php')) {
            $this->license_api_url = get_option('ra_kr_api_url', home_url('/wp-json/kr/v1/verify'));
        } else {
            $this->license_api_url = 'https://vase-domena.cz/wp-json/kr/v1/verify'; // Změňte na vaši doménu
        }
        
        // Admin menu
        add_action('admin_menu', array($this, 'add_license_menu'), 15);
        
        // Check license on admin init
        add_action('admin_init', array($this, 'check_license_status'));
        
        // Admin notices
        add_action('admin_notices', array($this, 'license_notices'));
        
        // AJAX handlers
        add_action('wp_ajax_kr_activate_license', array($this, 'activate_license'));
        add_action('wp_ajax_kr_deactivate_license', array($this, 'deactivate_license'));
        add_action('wp_ajax_kr_start_demo', array($this, 'start_demo'));
    }
    
    /**
     * Add license menu
     */
    public function add_license_menu() {
        add_submenu_page(
            'kadernice-rezervace',
            __('Licence', 'kadernice-rezervace'),
            __('🔑 Licence', 'kadernice-rezervace'),
            'manage_options',
            'kr-license',
            array($this, 'render_license_page')
        );
    }
    
    /**
     * Check license status
     */
    public function check_license_status() {
        $license_data = $this->get_license_data();
        
        // Check if demo expired
        if ($license_data['type'] === 'demo' && $license_data['expires_at']) {
            if (strtotime($license_data['expires_at']) < time()) {
                // Demo expired - deactivate plugin features
                update_option('kr_license_status', 'expired');
                update_option('kr_features_enabled', 0);
            }
        }
        
        // Check if full license expired
        if ($license_data['type'] === 'full' && $license_data['expires_at']) {
            if (strtotime($license_data['expires_at']) < time()) {
                update_option('kr_license_status', 'expired');
            }
        }
    }
    
    /**
     * Get license data
     */
    public function get_license_data() {
        return array(
            'key' => get_option('kr_license_key', ''),
            'type' => get_option('kr_license_type', 'none'), // none, demo, full
            'status' => get_option('kr_license_status', 'inactive'), // inactive, active, expired
            'activated_at' => get_option('kr_license_activated_at', ''),
            'expires_at' => get_option('kr_license_expires_at', ''),
            'email' => get_option('kr_license_email', ''),
            'features_enabled' => get_option('kr_features_enabled', 0)
        );
    }
    
    /**
     * Start demo
     */
    public function start_demo() {
        check_ajax_referer('kr-license-nonce', 'nonce');
        
        // Check if demo already used
        if (get_option('kr_demo_used', 0)) {
            wp_send_json_error(array('message' => 'Demo verze již byla použita. Pro pokračování zakupte plnou licenci.'));
        }
        
        // Generate demo key
        $demo_key = 'DEMO-' . strtoupper(substr(md5(uniqid()), 0, 12));
        
        // Set demo license
        update_option('kr_license_key', $demo_key);
        update_option('kr_license_type', 'demo');
        update_option('kr_license_status', 'active');
        update_option('kr_license_activated_at', current_time('mysql'));
        update_option('kr_license_expires_at', date('Y-m-d H:i:s', strtotime('+14 days')));
        update_option('kr_features_enabled', 1);
        update_option('kr_demo_used', 1);
        
        wp_send_json_success(array(
            'message' => 'Demo verze aktivována na 14 dní!',
            'expires_at' => date('d.m.Y', strtotime('+14 days'))
        ));
    }
    
    /**
     * Activate license
     */
    public function activate_license() {
        check_ajax_referer('kr-license-nonce', 'nonce');
        
        $license_key = sanitize_text_field($_POST['license_key']);
        $email = sanitize_email($_POST['email']);
        
        if (empty($license_key) || empty($email)) {
            wp_send_json_error(array('message' => 'Vyplňte licenční klíč a email.'));
        }
        
        // Verify license with API
        $response = wp_remote_post($this->license_api_url, array(
            'body' => array(
                'action' => 'verify',
                'license_key' => $license_key,
                'email' => $email,
                'domain' => home_url()
            ),
            'timeout' => 15
        ));
        
        if (is_wp_error($response)) {
            // Fallback - activate offline
            $this->activate_offline($license_key, $email);
            wp_send_json_success(array('message' => 'Licence aktivována (offline režim).'));
            return;
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if ($body && $body['success']) {
            // Activate license
            update_option('kr_license_key', $license_key);
            update_option('kr_license_type', 'full');
            update_option('kr_license_status', 'active');
            update_option('kr_license_activated_at', current_time('mysql'));
            update_option('kr_license_expires_at', $body['expires_at']);
            update_option('kr_license_email', $email);
            update_option('kr_features_enabled', 1);
            
            wp_send_json_success(array('message' => 'Licence úspěšně aktivována!'));
        } else {
            wp_send_json_error(array('message' => $body['message'] ?? 'Neplatný licenční klíč.'));
        }
    }
    
    /**
     * Activate offline (fallback)
     */
    private function activate_offline($license_key, $email) {
        // Simple validation
        if (preg_match('/^KR-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}$/', $license_key)) {
            update_option('kr_license_key', $license_key);
            update_option('kr_license_type', 'full');
            update_option('kr_license_status', 'active');
            update_option('kr_license_activated_at', current_time('mysql'));
            update_option('kr_license_expires_at', date('Y-m-d H:i:s', strtotime('+1 year')));
            update_option('kr_license_email', $email);
            update_option('kr_features_enabled', 1);
        }
    }
    
    /**
     * Deactivate license
     */
    public function deactivate_license() {
        check_ajax_referer('kr-license-nonce', 'nonce');
        
        delete_option('kr_license_key');
        delete_option('kr_license_type');
        delete_option('kr_license_status');
        delete_option('kr_license_activated_at');
        delete_option('kr_license_expires_at');
        delete_option('kr_license_email');
        update_option('kr_features_enabled', 0);
        
        wp_send_json_success(array('message' => 'Licence deaktivována.'));
    }
    
    /**
     * License notices
     */
    public function license_notices() {
        $license_data = $this->get_license_data();
        
        // Demo expiring soon
        if ($license_data['type'] === 'demo' && $license_data['status'] === 'active') {
            $days_left = floor((strtotime($license_data['expires_at']) - time()) / 86400);
            
            if ($days_left <= 3) {
                ?>
                <div class="notice notice-warning">
                    <p>
                        <strong>⏰ Demo verze vyprší za <?php echo $days_left; ?> <?php echo $days_left === 1 ? 'den' : 'dny'; ?>!</strong>
                        <a href="<?php echo admin_url('admin.php?page=kr-license'); ?>" class="button button-primary" style="margin-left: 10px;">
                            Zakoupit plnou licenci
                        </a>
                    </p>
                </div>
                <?php
            }
        }
        
        // License expired
        if ($license_data['status'] === 'expired') {
            ?>
            <div class="notice notice-error">
                <p>
                    <strong>🔒 Licence vypršela!</strong> 
                    Plugin je deaktivován. 
                    <a href="<?php echo admin_url('admin.php?page=kr-license'); ?>" class="button button-primary" style="margin-left: 10px;">
                        Obnovit licenci
                    </a>
                </p>
            </div>
            <?php
        }
    }
    
    /**
     * Render license page
     */
    public function render_license_page() {
        $license_data = $this->get_license_data();
        include KR_PLUGIN_DIR . 'admin/pages/license.php';
    }
    
    /**
     * Check if features are enabled
     */
    public static function are_features_enabled() {
        return (bool) get_option('kr_features_enabled', 0);
    }
    
    /**
     * Get pricing info from rezervace-admin if available
     */
    public static function get_pricing_info() {
        $is_admin_plugin_active = is_plugin_active('rezervace-admin/rezervace-admin.php');
        
        if ($is_admin_plugin_active) {
            return array(
                'price' => get_option('ra_kr_price', 2990),
                'buy_url' => get_option('ra_kr_buy_url', home_url('/koupit-kadernice-rezervace')),
                'bank_account' => get_option('ra_kr_bank_account', ''),
                'formatted_price' => number_format(get_option('ra_kr_price', 2990), 0, ',', ' ')
            );
        }
        
        return array(
            'price' => 2990,
            'buy_url' => 'https://vase-domena.cz/koupit-kadernice-rezervace',
            'bank_account' => '',
            'formatted_price' => '2 990'
        );
    }
}

// Initialize
new KR_License();