<?php
/**
 * Integration with Rezervace Admin plugin
 * Connects both plugins to use shared database and admin interface
 */

if (!defined('ABSPATH')) exit;

class KR_Integration {
    
    private $shared_table = 'ra_licenses'; // Shared table from Rezervace Admin
    
    public function __construct() {
        // Check if Rezervace Admin is active
        add_action('admin_init', array($this, 'check_admin_plugin'));
        
        // Use shared database if Rezervace Admin is active
        add_filter('kr_orders_table_name', array($this, 'use_shared_table'));
        
        // Add integration notice
        add_action('admin_notices', array($this, 'integration_notice'));
    }
    
    /**
     * Check if Rezervace Admin plugin is active
     */
    public function check_admin_plugin() {
        if ($this->is_admin_plugin_active()) {
            // Admin plugin is active - use shared system
            update_option('kr_use_shared_admin', 1);
        } else {
            update_option('kr_use_shared_admin', 0);
        }
    }
    
    /**
     * Check if Rezervace Admin is active
     */
    private function is_admin_plugin_active() {
        return is_plugin_active('rezervace-admin/rezervace-admin.php') || 
               class_exists('RA_Shop');
    }
    
    /**
     * Use shared table from Rezervace Admin
     */
    public function use_shared_table($table_name) {
        if (get_option('kr_use_shared_admin', 0)) {
            global $wpdb;
            return $wpdb->prefix . $this->shared_table;
        }
        return $table_name;
    }
    
    /**
     * Show integration notice
     */
    public function integration_notice() {
        $screen = get_current_screen();
        
        // Only show on plugin pages
        if (strpos($screen->id, 'kadernice-rezervace') === false && 
            strpos($screen->id, 'kr-') === false) {
            return;
        }
        
        if ($this->is_admin_plugin_active()) {
            ?>
            <div class="notice notice-success">
                <p>
                    <strong>📋 Administrace objednávek v kadeřnictví</strong>
                    <a href="<?php echo admin_url('admin.php?page=kr-bookings'); ?>" style="margin-left: 12px;">
                        Zobrazit rezervace →
                    </a>
                </p>
            </div>
            <?php
        } else {
            ?>
            <div class="notice notice-info">
                <p>
                    <strong>💡 Tip:</strong> 
                    Pro centrální správu všech objednávek aktivujte plugin <strong>Rezervace Admin</strong>.
                </p>
            </div>
            <?php
        }
    }
    
    /**
     * Sync order to shared database
     */
    public static function sync_order_to_admin($order_id, $order_data) {
        if (!get_option('kr_use_shared_admin', 0)) {
            return; // Not using shared admin
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'ra_licenses';
        
        // Check if order already exists
        $exists = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM $table_name WHERE email = %s AND product_name = %s",
            $order_data['email'],
            'Kadeřnice Rezervace'
        ));
        
        if ($exists) {
            return $exists; // Already synced
        }
        
        // Insert into shared table
        $wpdb->insert(
            $table_name,
            array(
                'name' => $order_data['name'],
                'email' => $order_data['email'],
                'phone' => $order_data['phone'],
                'company' => isset($order_data['company']) ? $order_data['company'] : '',
                'status' => 'čeká na platbu',
                'product_name' => 'Kadeřnice Rezervace',
                'price' => $order_data['price'],
                'created_at' => current_time('mysql')
            )
        );
        
        return $wpdb->insert_id;
    }
    
    /**
     * Get shared admin URL
     */
    public static function get_admin_url() {
        if (get_option('kr_use_shared_admin', 0)) {
            return admin_url('admin.php?page=rezervace-admin');
        }
        return admin_url('admin.php?page=kr-shop');
    }
}

// Initialize integration
new KR_Integration();