<?php
/**
 * Database operations handler
 */

if (!defined('ABSPATH')) {
    exit;
}

class KR_Database {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Get table name with prefix
     */
    public static function get_table($table) {
        global $wpdb;
        return $wpdb->prefix . 'kr_' . $table;
    }
    
    /**
     * Get all services
     */
    public static function get_services($active_only = true) {
        global $wpdb;
        $table = self::get_table('services');
        
        $where = $active_only ? "WHERE is_active = 1" : "";
        $sql = "SELECT * FROM $table $where ORDER BY name ASC";
        
        return $wpdb->get_results($sql);
    }
    
    /**
     * Get service by ID
     */
    public static function get_service($id) {
        global $wpdb;
        $table = self::get_table('services');
        
        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE id = %d",
            $id
        ));
    }
    
    /**
     * Get bookings
     */
    public static function get_bookings($args = array()) {
        global $wpdb;
        $table = self::get_table('bookings');
        
        $defaults = array(
            'date_from' => null,
            'date_to' => null,
            'status' => null,
            'limit' => 100,
            'offset' => 0,
            'orderby' => 'booking_date',
            'order' => 'ASC'
        );
        
        $args = wp_parse_args($args, $defaults);
        
        $where = array('1=1');
        
        if ($args['date_from']) {
            $where[] = $wpdb->prepare("booking_date >= %s", $args['date_from']);
        }
        
        if ($args['date_to']) {
            $where[] = $wpdb->prepare("booking_date <= %s", $args['date_to']);
        }
        
        if ($args['status']) {
            $where[] = $wpdb->prepare("status = %s", $args['status']);
        }
        
        $where_sql = implode(' AND ', $where);
        $orderby = sanitize_sql_orderby($args['orderby'] . ' ' . $args['order']);
        
        $sql = "SELECT b.*, s.name as service_name, s.duration, s.price 
                FROM $table b
                LEFT JOIN " . self::get_table('services') . " s ON b.service_id = s.id
                WHERE $where_sql
                ORDER BY $orderby
                LIMIT %d OFFSET %d";
        
        return $wpdb->get_results($wpdb->prepare($sql, $args['limit'], $args['offset']));
    }
    
    /**
     * Get booking by ID
     */
    public static function get_booking($id) {
        global $wpdb;
        $table = self::get_table('bookings');
        
        return $wpdb->get_row($wpdb->prepare(
            "SELECT b.*, s.name as service_name, s.duration, s.price 
             FROM $table b
             LEFT JOIN " . self::get_table('services') . " s ON b.service_id = s.id
             WHERE b.id = %d",
            $id
        ));
    }
    
    /**
     * Create booking
     */
    public static function create_booking($data) {
        global $wpdb;
        $table = self::get_table('bookings');
        
        // Ensure multi-service columns exist
        self::maybe_add_multi_service_columns();
        
        // Ensure customer_initials column exists
        self::maybe_add_customer_initials_column();
        
        $defaults = array(
            'status' => 'pending',
            'is_multi_service' => 0,
            'multi_service_group' => null,
            'created_at' => current_time('mysql')
        );
        
        $data = wp_parse_args($data, $defaults);
        
        $result = $wpdb->insert($table, $data);
        
        if ($result) {
            return $wpdb->insert_id;
        }
        
        return false;
    }
    
    /**
     * Add multi-service columns if they don't exist
     */
    private static function maybe_add_multi_service_columns() {
        global $wpdb;
        $table = self::get_table('bookings');
        
        // Check if columns exist
        $columns = $wpdb->get_results("SHOW COLUMNS FROM $table LIKE 'is_multi_service'");
        
        if (empty($columns)) {
            // Add is_multi_service column
            $wpdb->query("ALTER TABLE $table ADD COLUMN is_multi_service tinyint(1) NOT NULL DEFAULT 0 AFTER status");
        }
        
        $columns = $wpdb->get_results("SHOW COLUMNS FROM $table LIKE 'multi_service_group'");
        
        if (empty($columns)) {
            // Add multi_service_group column
            $wpdb->query("ALTER TABLE $table ADD COLUMN multi_service_group bigint(20) DEFAULT NULL AFTER is_multi_service");
            $wpdb->query("ALTER TABLE $table ADD KEY multi_service_group (multi_service_group)");
        }
    }
    
    /**
     * Add customer_initials column if it doesn't exist
     */
    private static function maybe_add_customer_initials_column() {
        global $wpdb;
        $table = self::get_table('bookings');
        
        // Check if column exists
        $columns = $wpdb->get_results("SHOW COLUMNS FROM $table LIKE 'customer_initials'");
        
        if (empty($columns)) {
            // Add customer_initials column after customer_phone
            $wpdb->query("ALTER TABLE $table ADD COLUMN customer_initials varchar(10) DEFAULT NULL AFTER customer_phone");
        }
    }
    
    /**
     * Update booking
     */
    public static function update_booking($id, $data) {
        global $wpdb;
        $table = self::get_table('bookings');
        
        $data['updated_at'] = current_time('mysql');
        
        return $wpdb->update($table, $data, array('id' => $id));
    }
    
    /**
     * Delete booking
     */
    public static function delete_booking($id) {
        global $wpdb;
        $table = self::get_table('bookings');
        
        return $wpdb->delete($table, array('id' => $id));
    }
    
    /**
     * Get booked slots for a date
     */
    public static function get_booked_slots($date, $service_id = null) {
        global $wpdb;
        $table = self::get_table('bookings');
        
        $where = array(
            $wpdb->prepare("booking_date = %s", $date),
            "status IN ('pending', 'confirmed')"
        );
        
        if ($service_id) {
            $where[] = $wpdb->prepare("service_id = %d", $service_id);
        }
        
        $where_sql = implode(' AND ', $where);
        
        $sql = "SELECT booking_time, service_id, s.duration
                FROM $table b
                LEFT JOIN " . self::get_table('services') . " s ON b.service_id = s.id
                WHERE $where_sql
                ORDER BY booking_time ASC";
        
        return $wpdb->get_results($sql);
    }
    
    /**
     * Get time slots for a day
     */
    public static function get_time_slots($day_of_week) {
        global $wpdb;
        $table = self::get_table('time_slots');
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table WHERE day_of_week = %d AND is_active = 1",
            $day_of_week
        ));
    }
    
    /**
     * Check if date is blocked
     */
    public static function is_date_blocked($date) {
        global $wpdb;
        $table = self::get_table('blocked_dates');
        
        $count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table WHERE blocked_date = %s",
            $date
        ));
        
        return $count > 0;
    }
    
    /**
     * Get blocked dates
     */
    public static function get_blocked_dates($from = null, $to = null) {
        global $wpdb;
        $table = self::get_table('blocked_dates');
        
        $where = array();
        
        if ($from) {
            $where[] = $wpdb->prepare("blocked_date >= %s", $from);
        }
        
        if ($to) {
            $where[] = $wpdb->prepare("blocked_date <= %s", $to);
        }
        
        $where_sql = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';
        
        return $wpdb->get_results("SELECT * FROM $table $where_sql ORDER BY blocked_date ASC");
    }
}