<?php

/**
 * Bookings management
 */

if (!defined('ABSPATH')) {
    exit;
}

class KR_Bookings
{

    /**
     * Create new booking
     */
    public static function create($data)
    {
        // Validate data
        $errors = self::validate_booking_data($data);

        if (!empty($errors)) {
            return array('success' => false, 'errors' => $errors);
        }

        // Check if slot is available
        if (!self::is_slot_available($data['booking_date'], $data['booking_time'], $data['service_id'])) {
            return array(
                'success' => false,
                'errors' => array(__('Tento termín již není k dispozici.', 'kadernice-rezervace'))
            );
        }

        // Create booking
        $booking_id = KR_Database::create_booking($data);

        if ($booking_id) {
            // Send email notifications
            KR_Email::send_booking_confirmation($booking_id);

            return array(
                'success' => true,
                'booking_id' => $booking_id,
                'message' => __('Rezervace byla úspěšně vytvořena!', 'kadernice-rezervace')
            );
        }

        return array(
            'success' => false,
            'errors' => array(__('Nepodařilo se vytvořit rezervaci. Zkuste to prosím znovu.', 'kadernice-rezervace'))
        );
    }

    /**
     * Validate booking data
     */
    private static function validate_booking_data($data)
    {
        $errors = array();

        // Required fields
        $required = array('service_id', 'customer_name', 'customer_email', 'customer_phone', 'booking_date', 'booking_time');

        foreach ($required as $field) {
            if (empty($data[$field])) {
                $errors[] = sprintf(__('Pole %s je povinné.', 'kadernice-rezervace'), $field);
            }
        }

        // Validate email
        if (!empty($data['customer_email'])) {
            $email = $data['customer_email'];
            $is_valid_email = false;
            if (function_exists('is_email')) {
                $is_valid_email = call_user_func('is_email', $email);
            } else {
                $is_valid_email = filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
            }
            if (!$is_valid_email) {
                $errors[] = __('Neplatná emailová adresa.', 'kadernice-rezervace');
            }
        }

        // Validate date
        if (!empty($data['booking_date'])) {
            $date = DateTime::createFromFormat('Y-m-d', $data['booking_date']);
            if (!$date || $date < new DateTime('today')) {
                $errors[] = __('Neplatné datum rezervace.', 'kadernice-rezervace');
            }
        }

        // Validate service exists
        if (!empty($data['service_id'])) {
            $service = KR_Database::get_service($data['service_id']);
            if (!$service) {
                $errors[] = __('Vybraná služba neexistuje.', 'kadernice-rezervace');
            }
        }

        return $errors;
    }

    /**
     * Check if time slot is available
     */
    public static function is_slot_available($date, $time, $service_id)
    {
        // Check if date is blocked
        if (KR_Database::is_date_blocked($date)) {
            return false;
        }

        // Check minimum booking notice
        $min_notice = get_option('kr_min_booking_notice', 24);
        $booking_datetime = new DateTime($date . ' ' . $time);
        $min_datetime = new DateTime('+' . $min_notice . ' hours');

        if ($booking_datetime < $min_datetime) {
            return false;
        }

        // Get service duration
        $service = KR_Database::get_service($service_id);
        if (!$service) {
            return false;
        }

        // Get booked slots for this date
        $booked_slots = KR_Database::get_booked_slots($date);

        // Check for conflicts
        $requested_start = new DateTime($date . ' ' . $time);
        $requested_end = clone $requested_start;
        $requested_end->modify('+' . $service->duration . ' minutes');

        foreach ($booked_slots as $slot) {
            $slot_start = new DateTime($date . ' ' . $slot->booking_time);
            $slot_end = clone $slot_start;
            $slot_end->modify('+' . $slot->duration . ' minutes');

            // Check if times overlap
            if ($requested_start < $slot_end && $requested_end > $slot_start) {
                return false;
            }
        }

        return true;
    }

    /**
     * Get available time slots for a date
     */
    public static function get_available_slots($date, $service_id)
    {
        $date_obj = new DateTime($date);
        $day_of_week = $date_obj->format('N'); // 1 (Monday) to 7 (Sunday)

        // Get working hours for this day
        $time_slots = KR_Database::get_time_slots($day_of_week);

        if (empty($time_slots)) {
            return array('available' => array(), 'booked' => array());
        }

        $service = KR_Database::get_service($service_id);
        if (!$service) {
            return array('available' => array(), 'booked' => array());
        }

        $available = array();
        $booked = array();
        $interval = get_option('kr_booking_interval', 30);

        foreach ($time_slots as $slot) {
            $start = new DateTime($date . ' ' . $slot->start_time);
            $end = new DateTime($date . ' ' . $slot->end_time);

            // Generate time slots
            while ($start < $end) {
                $time = $start->format('H:i');

                // Check if slot can fit the service duration
                $slot_end = clone $start;
                $slot_end->modify('+' . $service->duration . ' minutes');

                if ($slot_end <= $end) {
                    if (self::is_slot_available($date, $time, $service_id)) {
                        $available[] = $time;
                    } else {
                        $booked[] = $time;
                    }
                }

                $start->modify('+' . $interval . ' minutes');
            }
        }

        return array(
            'available' => $available,
            'booked' => $booked
        );
    }

    /**
     * Get available dates
     */
    public static function get_available_dates($service_id, $from_date = null, $days = 60)
    {
        if (!$from_date) {
            $from_date = new DateTime('today');
        } else {
            $from_date = new DateTime($from_date);
        }

        $available_dates = array();
        $blocked_dates = KR_Database::get_blocked_dates(
            $from_date->format('Y-m-d'),
            (clone $from_date)->modify('+' . $days . ' days')->format('Y-m-d')
        );

        $blocked_array = array();
        foreach ($blocked_dates as $blocked) {
            $blocked_array[] = $blocked->blocked_date;
        }

        for ($i = 0; $i < $days; $i++) {
            $date = clone $from_date;
            $date->modify('+' . $i . ' days');
            $date_str = $date->format('Y-m-d');

            // Skip blocked dates
            if (in_array($date_str, $blocked_array)) {
                continue;
            }

            // Check if day has working hours
            $day_of_week = $date->format('N');
            $time_slots = KR_Database::get_time_slots($day_of_week);

            if (!empty($time_slots)) {
                // Check if at least one slot is available
                $slots = self::get_available_slots($date_str, $service_id);
                if (!empty($slots)) {
                    $available_dates[] = $date_str;
                }
            }
        }

        return $available_dates;
    }

    /**
     * Update booking status
     */
    public static function update_status($booking_id, $status)
    {
        $valid_statuses = array('pending', 'confirmed', 'cancelled', 'completed');

        if (!in_array($status, $valid_statuses)) {
            return false;
        }

        return KR_Database::update_booking($booking_id, array('status' => $status));
    }

    /**
     * Create booking with multiple services
     */
    public static function create_multi_service($data)
    {
        // Validate data
        $errors = self::validate_multi_service_booking_data($data);

        if (!empty($errors)) {
            return array('success' => false, 'errors' => $errors);
        }

        // Calculate total duration
        $total_duration = 0;
        $service_names = array();
        $total_price = 0;

        foreach ($data['service_ids'] as $service_id) {
            $service = KR_Database::get_service($service_id);
            if ($service) {
                $total_duration += $service->duration;
                $service_names[] = $service->name;
                $total_price += $service->price;
            }
        }

        // Check if slot is available for total duration
        if (!self::is_slot_available_for_duration($data['booking_date'], $data['booking_time'], $total_duration)) {
            return array(
                'success' => false,
                'errors' => array(__('Tento termín již není k dispozici pro vybrané služby.', 'kadernice-rezervace'))
            );
        }

        // Create bookings for each service
        $booking_ids = array();
        $main_booking_id = null;

        foreach ($data['service_ids'] as $index => $service_id) {
            $booking_data = array(
                'service_id' => $service_id,
                'customer_name' => $data['customer_name'],
                'customer_email' => $data['customer_email'],
                'customer_phone' => $data['customer_phone'],
                'booking_date' => $data['booking_date'],
                'booking_time' => $data['booking_time'],
                'notes' => $data['notes'],
                'is_multi_service' => 1,
                'multi_service_group' => null // Will be set after first booking
            );

            $booking_id = KR_Database::create_booking($booking_data);

            if ($booking_id) {
                $booking_ids[] = $booking_id;

                // First booking is the main one
                if ($main_booking_id === null) {
                    $main_booking_id = $booking_id;

                    // Update all bookings with the group ID
                    foreach ($booking_ids as $id) {
                        KR_Database::update_booking($id, array('multi_service_group' => $main_booking_id));
                    }
                } else {
                    // Update with group ID
                    KR_Database::update_booking($booking_id, array('multi_service_group' => $main_booking_id));
                }
            }
        }

        if (!empty($booking_ids)) {
            // Send email notification for main booking
            KR_Email::send_multi_service_confirmation($main_booking_id, $service_names, $total_duration, $total_price);

            return array(
                'success' => true,
                'booking_ids' => $booking_ids,
                'main_booking_id' => $main_booking_id,
                'message' => __('Rezervace byla úspěšně vytvořena!', 'kadernice-rezervace')
            );
        }

        return array(
            'success' => false,
            'errors' => array(__('Nepodařilo se vytvořit rezervaci. Zkuste to prosím znovu.', 'kadernice-rezervace'))
        );
    }

    /**
     * Validate multi-service booking data
     */
    private static function validate_multi_service_booking_data($data)
    {
        $errors = array();

        // Required fields
        if (empty($data['service_ids']) || !is_array($data['service_ids'])) {
            $errors[] = __('Vyberte alespoň jednu službu.', 'kadernice-rezervace');
        }

        $required = array('customer_name', 'customer_email', 'customer_phone', 'booking_date', 'booking_time');

        foreach ($required as $field) {
            if (empty($data[$field])) {
                $errors[] = sprintf(__('Pole %s je povinné.', 'kadernice-rezervace'), $field);
            }
        }

        // Validate email
        if (!empty($data['customer_email'])) {
            $email = $data['customer_email'];
            $is_valid_email = false;
            if (function_exists('is_email')) {
                $is_valid_email = call_user_func('is_email', $email);
            } else {
                $is_valid_email = filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
            }
            if (!$is_valid_email) {
                $errors[] = __('Neplatná emailová adresa.', 'kadernice-rezervace');
            }
        }

        // Validate date
        if (!empty($data['booking_date'])) {
            $date = DateTime::createFromFormat('Y-m-d', $data['booking_date']);
            if (!$date || $date < new DateTime('today')) {
                $errors[] = __('Neplatné datum rezervace.', 'kadernice-rezervace');
            }
        }

        // Validate services exist
        if (!empty($data['service_ids'])) {
            foreach ($data['service_ids'] as $service_id) {
                $service = KR_Database::get_service($service_id);
                if (!$service) {
                    $errors[] = sprintf(__('Služba s ID %d neexistuje.', 'kadernice-rezervace'), $service_id);
                }
            }
        }

        return $errors;
    }

    /**
     * Check if time slot is available for specific duration
     */
    public static function is_slot_available_for_duration($date, $time, $duration)
    {
        // Check if date is blocked
        if (KR_Database::is_date_blocked($date)) {
            return false;
        }

        // Check minimum booking notice
        $min_notice = get_option('kr_min_booking_notice', 24);
        $booking_datetime = new DateTime($date . ' ' . $time);
        $min_datetime = new DateTime('+' . $min_notice . ' hours');

        if ($booking_datetime < $min_datetime) {
            return false;
        }

        // Get booked slots for this date
        $booked_slots = KR_Database::get_booked_slots($date);

        // Check for conflicts
        $requested_start = new DateTime($date . ' ' . $time);
        $requested_end = clone $requested_start;
        $requested_end->modify('+' . $duration . ' minutes');

        foreach ($booked_slots as $slot) {
            $slot_start = new DateTime($date . ' ' . $slot->booking_time);
            $slot_end = clone $slot_start;
            $slot_end->modify('+' . $slot->duration . ' minutes');

            // Check if times overlap
            if ($requested_start < $slot_end && $requested_end > $slot_start) {
                return false;
            }
        }

        return true;
    }

    /**
     * Get available time slots for multiple services
     */
    public static function get_available_slots_for_services($date, $service_ids, $total_duration)
    {
        $date_obj = new DateTime($date);
        $day_of_week = $date_obj->format('N');

        // Get working hours for this day
        $time_slots = KR_Database::get_time_slots($day_of_week);

        if (empty($time_slots)) {
            return array('available' => array(), 'booked' => array());
        }

        $available = array();
        $booked = array();
        $interval = get_option('kr_booking_interval', 30);

        foreach ($time_slots as $slot) {
            $start = new DateTime($date . ' ' . $slot->start_time);
            $end = new DateTime($date . ' ' . $slot->end_time);

            // Generate time slots
            while ($start < $end) {
                $time = $start->format('H:i');

                // Check if slot can fit the total duration
                $slot_end = clone $start;
                $slot_end->modify('+' . $total_duration . ' minutes');

                if ($slot_end <= $end) {
                    if (self::is_slot_available_for_duration($date, $time, $total_duration)) {
                        $available[] = $time;
                    } else {
                        $booked[] = $time;
                    }
                }

                $start->modify('+' . $interval . ' minutes');
            }
        }

        return array(
            'available' => $available,
            'booked' => $booked
        );
    }

    /**
     * Get available dates for multiple services
     */
    public static function get_available_dates_for_services($service_ids, $total_duration, $from_date = null, $days = 60)
    {
        if (!$from_date) {
            $from_date = new DateTime('today');
        } else {
            $from_date = new DateTime($from_date);
        }

        $available_dates = array();
        $blocked_dates = KR_Database::get_blocked_dates(
            $from_date->format('Y-m-d'),
            (clone $from_date)->modify('+' . $days . ' days')->format('Y-m-d')
        );

        $blocked_array = array();
        foreach ($blocked_dates as $blocked) {
            $blocked_array[] = $blocked->blocked_date;
        }

        for ($i = 0; $i < $days; $i++) {
            $date = clone $from_date;
            $date->modify('+' . $i . ' days');
            $date_str = $date->format('Y-m-d');

            // Skip blocked dates
            if (in_array($date_str, $blocked_array)) {
                continue;
            }

            // Check if day has working hours
            $day_of_week = $date->format('N');
            $time_slots = KR_Database::get_time_slots($day_of_week);

            if (!empty($time_slots)) {
                // Check if at least one slot is available
                $slots = self::get_available_slots_for_services($date_str, $service_ids, $total_duration);
                if (!empty($slots)) {
                    $available_dates[] = $date_str;
                }
            }
        }

        return $available_dates;
    }
}
