<?php
/**
 * AJAX handlers
 */

if (!defined('ABSPATH')) {
    require_once(dirname(__FILE__, 4) . '/wp-load.php');
    exit;
}

class KR_Ajax {

    public function __construct() {
        if (!function_exists('add_action')) {
            return;
        }

        // Frontend AJAX actions
        add_action('wp_ajax_kr_get_services', array($this, 'get_services'));
        add_action('wp_ajax_nopriv_kr_get_services', array($this, 'get_services'));
        add_action('wp_ajax_kr_get_available_dates', array($this, 'get_available_dates'));
        add_action('wp_ajax_nopriv_kr_get_available_dates', array($this, 'get_available_dates'));
        add_action('wp_ajax_kr_get_available_times', array($this, 'get_available_times'));
        add_action('wp_ajax_nopriv_kr_get_available_times', array($this, 'get_available_times'));
        add_action('wp_ajax_kr_create_booking', array($this, 'create_booking'));
        add_action('wp_ajax_nopriv_kr_create_booking', array($this, 'create_booking'));

        // Admin AJAX actions
        add_action('wp_ajax_kr_update_booking_status', array($this, 'update_booking_status'));
        add_action('wp_ajax_kr_delete_booking', array($this, 'delete_booking'));
        add_action('wp_ajax_kr_get_booking_detail', array($this, 'get_booking_detail'));
        add_action('wp_ajax_kr_send_customer_email', array($this, 'send_customer_email'));
    }

    // ===============================
    // FRONTEND METHODS
    // ===============================

    public function get_services() {
        check_ajax_referer('kr-ajax-nonce', 'nonce');

        $services = KR_Services::get_active_services();
        $response = array();

        foreach ($services as $service) {
            $response[] = array(
                'id' => $service->id,
                'name' => $service->name,
                'description' => $service->description,
                'duration' => $service->duration,
                'duration_formatted' => KR_Services::format_duration($service->duration),
                'price' => $service->price,
                'price_formatted' => KR_Services::format_price($service->price),
                'color' => $service->color
            );
        }

        wp_send_json_success($response);
    }

    public function get_available_dates() {
        check_ajax_referer('kr-ajax-nonce', 'nonce');

        $service_ids = isset($_POST['service_ids']) ? $_POST['service_ids'] : array();
        $total_duration = isset($_POST['total_duration']) ? intval($_POST['total_duration']) : 0;
        $from_date = isset($_POST['from_date']) ? sanitize_text_field($_POST['from_date']) : null;

        if (empty($service_ids) && isset($_POST['service_id'])) {
            $service_ids = array(intval($_POST['service_id']));
        }

        if (empty($service_ids)) {
            wp_send_json_error(array('message' => __('Vyberte službu', 'kadernice-rezervace')));
        }

        $service_ids = array_map('intval', (array)$service_ids);
        $dates = KR_Bookings::get_available_dates_for_services($service_ids, $total_duration, $from_date);

        wp_send_json_success($dates);
    }

    public function get_available_times() {
        check_ajax_referer('kr-ajax-nonce', 'nonce');

        $service_ids = isset($_POST['service_ids']) ? $_POST['service_ids'] : array();
        $total_duration = isset($_POST['total_duration']) ? intval($_POST['total_duration']) : 0;
        $date = isset($_POST['date']) ? sanitize_text_field($_POST['date']) : '';

        if (empty($service_ids) && isset($_POST['service_id'])) {
            $service_ids = array(intval($_POST['service_id']));
        }

        if (empty($service_ids) || !$date) {
            wp_send_json_error(array('message' => __('Neplatné parametry', 'kadernice-rezervace')));
        }

        $service_ids = array_map('intval', (array)$service_ids);
        $times = KR_Bookings::get_available_slots_for_services($date, $service_ids, $total_duration);

        wp_send_json_success($times);
    }

   public function create_booking() {
    check_ajax_referer('kr-ajax-nonce', 'nonce');

    $service_ids = isset($_POST['service_ids']) ? $_POST['service_ids'] : '';

    if (is_string($service_ids) && !empty($service_ids)) {
        $service_ids = array_filter(array_map('intval', explode(',', $service_ids)));
    } elseif (!is_array($service_ids)) {
        $service_ids = array();
    }

    if (empty($service_ids) && isset($_POST['service_id'])) {
        $service_ids = array(intval($_POST['service_id']));
    }

    if (empty($service_ids)) {
        wp_send_json_error(array('errors' => array(__('Vyberte alespoň jednu službu.', 'kadernice-rezervace'))));
        return;
    }

    $data = array(
        'customer_name'     => isset($_POST['customer_name']) ? sanitize_text_field($_POST['customer_name']) : '',
        'customer_email'    => isset($_POST['customer_email']) ? sanitize_email($_POST['customer_email']) : '',
        'customer_phone'    => isset($_POST['customer_phone']) ? sanitize_text_field($_POST['customer_phone']) : '',
        'customer_initials' => isset($_POST['customer_initials']) ? sanitize_text_field($_POST['customer_initials']) : '',
        'booking_date'      => isset($_POST['booking_date']) ? sanitize_text_field($_POST['booking_date']) : '',
        'booking_time'      => isset($_POST['booking_time']) ? sanitize_text_field($_POST['booking_time']) : '',
        'notes'             => isset($_POST['notes']) ? sanitize_textarea_field($_POST['notes']) : ''
    );

    if (count($service_ids) > 1) {
        $data['service_ids'] = $service_ids;
        $result = KR_Bookings::create_multi_service($data);
    } else {
        $data['service_id'] = $service_ids[0];
        $result = KR_Bookings::create($data);
    }

    if ($result['success']) {
        // >>> Tady se spustí odeslání e‑mailu
        if (!empty($result['booking_id'])) {
            KR_Email::send_booking_confirmation($result['booking_id']);
        }

        wp_send_json_success($result);
    } else {
        wp_send_json_error($result);
    }
}


    // ===============================
    // ADMIN METHODS
    // ===============================

    public function update_booking_status() {
        check_ajax_referer('kr-admin-nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Nemáte oprávnění', 'kadernice-rezervace')));
        }

        $booking_id = isset($_POST['booking_id']) ? intval($_POST['booking_id']) : 0;
        $status = isset($_POST['status']) ? sanitize_text_field($_POST['status']) : '';

        if (!$booking_id || !$status) {
            wp_send_json_error(array('message' => __('Neplatné parametry', 'kadernice-rezervace')));
        }

        $result = KR_Bookings::update_status($booking_id, $status);

        if ($result) {
            wp_send_json_success(array('message' => __('Status byl aktualizován', 'kadernice-rezervace')));
        } else {
            wp_send_json_error(array('message' => __('Nepodařilo se aktualizovat status', 'kadernice-rezervace')));
        }
    }

    public function delete_booking() {
        check_ajax_referer('kr-admin-nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Nemáte oprávnění', 'kadernice-rezervace')));
        }

        $booking_id = isset($_POST['booking_id']) ? intval($_POST['booking_id']) : 0;

        if (!$booking_id) {
            wp_send_json_error(array('message' => __('Neplatné ID rezervace', 'kadernice-rezervace')));
        }

        $result = KR_Database::delete_booking($booking_id);

        if ($result) {
            wp_send_json_success(array('message' => __('Rezervace byla smazána', 'kadernice-rezervace')));
        } else {
            wp_send_json_error(array('message' => __('Nepodařilo se smazat rezervaci', 'kadernice-rezervace')));
        }
    }

    public function get_booking_detail() {
        check_ajax_referer('kr-admin-nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Nemáte oprávnění', 'kadernice-rezervace')));
        }

        $booking_id = isset($_POST['booking_id']) ? intval($_POST['booking_id']) : 0;

        if (!$booking_id) {
            wp_send_json_error(array('message' => __('Neplatné ID rezervace', 'kadernice-rezervace')));
        }

        $booking = KR_Database::get_booking($booking_id);

        if (!$booking) {
            wp_send_json_error(array('message' => __('Rezervace nenalezena', 'kadernice-rezervace')));
        }

        $date = date_i18n(get_option('kr_date_format', 'd.m.Y'), strtotime($booking->booking_date));
        $time = date_i18n(get_option('kr_time_format', 'H:i'), strtotime($booking->booking_time));

        $statuses = array(
            'pending' => __('Čekající', 'kadernice-rezervace'),
            'confirmed' => __('Potvrzené', 'kadernice-rezervace'),
            'cancelled' => __('Zrušené', 'kadernice-rezervace'),
            'completed' => __('Dokončené', 'kadernice-rezervace')
        );

        ob_start();
        ?>
        <div class="kr-booking-info">
            <div class="kr-booking-info-row">
                <div class="kr-booking-info-label"><?php _e('ID rezervace:', 'kadernice-rezervace'); ?></div>
                <div class="kr-booking-info-value">#<?php echo esc_html($booking->id); ?></div>
            </div>
            <div class="kr-booking-info-row">
                <div class="kr-booking-info-label"><?php _e('Zákazník:', 'kadernice-rezervace'); ?></div>
                <div class="kr-booking-info-value"><strong><?php echo esc_html($booking->customer_name); ?></strong></div>
            </div>
            <div class="kr-booking-info-row">
                <div class="kr-booking-info-label"><?php _e('Email:', 'kadernice-rezervace'); ?></div>
                <div class="kr-booking-info-value">
                    <a href="mailto:<?php echo esc_attr($booking->customer_email); ?>"><?php echo esc_html($booking->customer_email); ?></a>
                </div>
            </div>
            <div class="kr-booking-info-row">
                <div class="kr-booking-info-label"><?php _e('Telefon:', 'kadernice-rezervace'); ?></div>
                <div class="kr-booking-info-value">
                    <a href="tel:<?php echo esc_attr($booking->customer_phone); ?>"><?php echo esc_html($booking->customer_phone); ?></a>
                </div>
            </div>
            <div class="kr-booking-info-row">
                <div class="kr-booking-info-label"><?php _e('Služba:', 'kadernice-rezervace'); ?></div>
                <div class="kr-booking-info-value"><?php echo esc_html($booking->service_name); ?></div>
            </div>
            <div class="kr-booking-info-row">
                <div class="kr-booking-info-label"><?php _e('Datum:', 'kadernice-rezervace'); ?></div>
                <div class="kr-booking-info-value"><?php echo esc_html($date); ?></div>
            </div>
            <div class="kr-booking-info-row">
                <div class="kr-booking-info-label"><?php _e('Čas:', 'kadernice-rezervace'); ?></div>
                <div class="kr-booking-info-value"><?php echo esc_html($time); ?></div>
            </div>
            <div class="kr-booking-info-row">
                <div class="kr-booking-info-label"><?php _e('Doba trvání:', 'kadernice-rezervace'); ?></div>
                <div class="kr-booking-info-value"><?php echo esc_html($booking->duration); ?> min</div>
            </div>
            <div class="kr-booking-info-row">
                <div class="kr-booking-info-label"><?php _e('Cena:', 'kadernice-rezervace'); ?></div>
                <div class="kr-booking-info-value"><?php echo number_format($booking->price, 0, ',', ' '); ?> <?php echo esc_html(get_option('kr_currency', 'Kč')); ?></div>
            </div>
            <div class="kr-booking-info-row">
                <div class="kr-booking-info-label"><?php _e('Status:', 'kadernice-rezervace'); ?></div>
                <div class="kr-booking-info-value"><?php echo esc_html($statuses[$booking->status] ?? $booking->status); ?></div>
            </div>
            <?php if ($booking->notes): ?>
            <div class="kr-booking-info-row">
                <div class="kr-booking-info-label"><?php _e('Poznámka:', 'kadernice-rezervace'); ?></div>
                <div class="kr-booking-info-value"><?php echo nl2br(esc_html($booking->notes)); ?></div>
            </div>
            <?php endif; ?>
            <div class="kr-booking-info-row">
                <div class="kr-booking-info-label"><?php _e('Vytvořeno:', 'kadernice-rezervace'); ?></div>
                <div class="kr-booking-info-value"><?php echo date_i18n('d.m.Y H:i', strtotime($booking->created_at)); ?></div>
            </div>
        </div>
        <?php
        $html = ob_get_clean();

        wp_send_json_success(array('html' => $html));
    }

   // ===============================
// SEND EMAIL (FIXED)
// ===============================

public function send_customer_email() {
    check_ajax_referer('kr-admin-nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => __('Nemáte oprávnění', 'kadernice-rezervace')));
    }

    $booking_id    = isset($_POST['booking_id']) ? intval($_POST['booking_id']) : 0;
    $email_to      = isset($_POST['email_to']) ? sanitize_email($_POST['email_to']) : '';
    $email_subject = isset($_POST['email_subject']) ? sanitize_text_field($_POST['email_subject']) : '';
    $email_message = isset($_POST['email_message']) ? sanitize_textarea_field($_POST['email_message']) : '';

    if (!$booking_id || !$email_to || !$email_subject || !$email_message) {
        wp_send_json_error(array('message' => __('Vyplňte všechna pole', 'kadernice-rezervace')));
    }

    // HTML zpráva
    $message  = '<html><body>';
    $message .= '<p>' . nl2br(esc_html($email_message)) . '</p>';
    $message .= '<hr>';
    $message .= '<p style="color:#666;font-size:12px;">';
    $message .= esc_html(get_option('kr_business_name')) . '<br>';
    if ($phone = get_option('kr_business_phone')) {
        $message .= __('Tel:', 'kadernice-rezervace') . ' ' . esc_html($phone) . '<br>';
    }
    if ($business_email = get_option('kr_business_email')) {
        $message .= __('Email:', 'kadernice-rezervace') . ' ' . esc_html($business_email);
    }
    $message .= '</p></body></html>';

    // Použijeme KR_Email místo wp_mail
    $sent = KR_Email::send_mail($email_to, $email_subject, $message);

    if ($sent) {
        wp_send_json_success(array('message' => __('Email byl odeslán', 'kadernice-rezervace')));
    } else {
        wp_send_json_error(array('message' => __('Nepodařilo se odeslat email', 'kadernice-rezervace')));
    }
}
}