<?php

/**
 * Plugin activation handler
 */

if (!defined('ABSPATH')) {
    exit;
}

class KR_Activator
{

    public static function activate()
    {
        // Create database tables
        self::create_tables();

        // Set default options
        self::set_default_options();

        // Create default services
        self::create_default_services();

        // Set transient for redirect to welcome page
        if (function_exists('set_transient')) {
            // call via call_user_func to avoid static analyzers complaining about undefined global functions
            call_user_func('set_transient', 'kr_activation_redirect', true, 30);
        } elseif (function_exists('add_option') && function_exists('update_option') && function_exists('get_option')) {
            // Fallback when transients are unavailable: store a simple option flag
            if (call_user_func('get_option', 'kr_activation_redirect') === false) {
                call_user_func('add_option', 'kr_activation_redirect', '1');
            } else {
                call_user_func('update_option', 'kr_activation_redirect', '1');
            }
        }

        // Flush rewrite rules (call safely to satisfy static analyzers)
        if (function_exists('flush_rewrite_rules')) {
            call_user_func('flush_rewrite_rules');
        }
    }

    private static function create_tables()
    {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();

        // Attempt to include WP upgrade functions; prefer ABSPATH when available, otherwise use relative path.
        $upgrade_path = null;
        if (defined('ABSPATH') && constant('ABSPATH')) {
            // use constant() to avoid static analyzers complaining about undefined constants
            $upgrade_path = rtrim(constant('ABSPATH'), '/\\') . '/wp-admin/includes/upgrade.php';
        }
        if (! $upgrade_path) {
            // Fallback: try to include the upgrade file using a relative path if WordPress constants are not available.
            // This avoids "Undefined constant 'ABSPATH'" errors when the file is parsed outside a WP context.
            $upgrade_path = __DIR__ . '/../../wp-admin/includes/upgrade.php';
        }
        if (file_exists($upgrade_path)) {
            require_once $upgrade_path;
        }

        // Services table
        $table_services = $wpdb->prefix . 'kr_services';
        $sql_services = "CREATE TABLE IF NOT EXISTS $table_services (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            name varchar(255) NOT NULL,
            description text,
            duration int(11) NOT NULL DEFAULT 30,
            price decimal(10,2) NOT NULL DEFAULT 0.00,
            color varchar(7) DEFAULT '#f89820',
            is_active tinyint(1) NOT NULL DEFAULT 1,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) $charset_collate;";

        // Ensure dbDelta is available; try to include WP upgrade functions if needed.
        if (! function_exists('dbDelta')) {
            if (defined('ABSPATH') && file_exists(rtrim(constant('ABSPATH'), '/\\') . '/wp-admin/includes/upgrade.php')) {
                require_once rtrim(constant('ABSPATH'), '/\\') . '/wp-admin/includes/upgrade.php';
            } elseif (file_exists(__DIR__ . '/../../wp-admin/includes/upgrade.php')) {
                require_once __DIR__ . '/../../wp-admin/includes/upgrade.php';
            }
        }

        if (function_exists('dbDelta')) {
            call_user_func('dbDelta', $sql_services);
        } else {
            // Fallback to direct query if dbDelta is not available.
            $wpdb->query($sql_services);
        }

        // Bookings table
        $table_bookings = $wpdb->prefix . 'kr_bookings';
        $sql_bookings = "CREATE TABLE IF NOT EXISTS $table_bookings (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            service_id bigint(20) NOT NULL,
            customer_name varchar(255) NOT NULL,
            customer_email varchar(255) NOT NULL,
            customer_phone varchar(50) NOT NULL,
            booking_date date NOT NULL,
            booking_time time NOT NULL,
            notes text,
            status varchar(20) NOT NULL DEFAULT 'pending',
            is_multi_service tinyint(1) NOT NULL DEFAULT 0,
            multi_service_group bigint(20) DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY service_id (service_id),
            KEY booking_date (booking_date),
            KEY status (status),
            KEY multi_service_group (multi_service_group)
        ) $charset_collate;";

        if (function_exists('dbDelta')) {
            call_user_func('dbDelta', $sql_bookings);
        } else {
            $wpdb->query($sql_bookings);
        }

        // Time slots table
        $table_slots = $wpdb->prefix . 'kr_time_slots';
        $sql_slots = "CREATE TABLE IF NOT EXISTS $table_slots (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            day_of_week tinyint(1) NOT NULL,
            start_time time NOT NULL,
            end_time time NOT NULL,
            is_active tinyint(1) NOT NULL DEFAULT 1,
            PRIMARY KEY (id),
            KEY day_of_week (day_of_week)
        ) $charset_collate;";

        if (function_exists('dbDelta')) {
            call_user_func('dbDelta', $sql_slots);
        } else {
            $wpdb->query($sql_slots);
        }

        // Blocked dates table
        $table_blocked = $wpdb->prefix . 'kr_blocked_dates';
        $sql_blocked = "CREATE TABLE IF NOT EXISTS $table_blocked (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            blocked_date date NOT NULL,
            reason varchar(255),
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY blocked_date (blocked_date)
        ) $charset_collate;";

        if (function_exists('dbDelta')) {
            call_user_func('dbDelta', $sql_blocked);
        } else {
            $wpdb->query($sql_blocked);
        }
    }

    private static function set_default_options()
    {
        $default_options = array(
            'kr_business_name' => (function_exists('get_bloginfo') ? call_user_func('get_bloginfo', 'name') : ''),
            'kr_business_email' => (function_exists('get_option') ? call_user_func('get_option', 'admin_email') : ''),
            'kr_business_phone' => '',
            'kr_booking_interval' => 30, // minutes
            'kr_advance_booking_days' => 60,
            'kr_min_booking_notice' => 24, // hours
            'kr_max_bookings_per_slot' => 1,
            'kr_enable_email_notifications' => 1,
            'kr_admin_email_on_booking' => 1,
            'kr_customer_email_on_booking' => 1,
            'kr_currency' => 'Kč',
            'kr_date_format' => 'd.m.Y',
            'kr_time_format' => 'H:i',
        );

        foreach ($default_options as $key => $value) {
            // If get_option exists, check whether the option is present.
            if (function_exists('get_option')) {
                $existing = call_user_func('get_option', $key);
                if ($existing === false) {
                    if (function_exists('add_option')) {
                        call_user_func('add_option', $key, $value);
                    } elseif (function_exists('update_option')) {
                        // fallback: update_option will add the option if it doesn't exist in some WP setups
                        call_user_func('update_option', $key, $value);
                    }
                }
            } else {
                // get_option not available (parsed outside WP context) — attempt to persist if add/update exist
                if (function_exists('add_option')) {
                    call_user_func('add_option', $key, $value);
                } elseif (function_exists('update_option')) {
                    call_user_func('update_option', $key, $value);
                }
            }
        }
    }

    private static function create_default_services()
    {
        global $wpdb;
        $table = $wpdb->prefix . 'kr_services';

        // Check if services already exist
        $count = $wpdb->get_var("SELECT COUNT(*) FROM $table");

        if ($count == 0) {
            $default_services = array(
                array(
                    'name' => 'Dámský střih',
                    'description' => 'Klasický dámský střih včetně mytí a foukání',
                    'duration' => 60,
                    'price' => 500.00,
                    'color' => '#f89820'
                ),
                array(
                    'name' => 'Pánský střih',
                    'description' => 'Pánský střih včetně mytí',
                    'duration' => 30,
                    'price' => 300.00,
                    'color' => '#2d4a45'
                ),
                array(
                    'name' => 'Barvení',
                    'description' => 'Kompletní barvení vlasů',
                    'duration' => 120,
                    'price' => 1200.00,
                    'color' => '#1e3a5a'
                ),
                array(
                    'name' => 'Melír',
                    'description' => 'Melírování vlasů',
                    'duration' => 90,
                    'price' => 900.00,
                    'color' => '#8b5cf6'
                ),
                array(
                    'name' => 'Trvalá',
                    'description' => 'Trvalá ondulace',
                    'duration' => 120,
                    'price' => 1000.00,
                    'color' => '#ec4899'
                ),
            );

            foreach ($default_services as $service) {
                $wpdb->insert($table, $service);
            }
        }

        // Create default time slots (Monday to Friday, 9:00-17:00)
        $table_slots = $wpdb->prefix . 'kr_time_slots';
        $count_slots = $wpdb->get_var("SELECT COUNT(*) FROM $table_slots");

        if ($count_slots == 0) {
            for ($day = 1; $day <= 5; $day++) { // Monday to Friday
                $wpdb->insert($table_slots, array(
                    'day_of_week' => $day,
                    'start_time' => '09:00:00',
                    'end_time' => '17:00:00',
                    'is_active' => 1
                ));
            }
        }
    }
}
