/**
 * Frontend JavaScript for Kadeřnice Rezervace
 */

(function($) {
    'use strict';
    
    const KR_Booking = {
        currentStep: 1,
        selectedServices: [], // Changed from selectedService to array
        selectedDate: null,
        selectedTime: null,
        currentMonth: new Date(),
        availableDates: [],
        
        init: function() {
            this.loadServices();
            this.bindEvents();
        },
        
        bindEvents: function() {
            const self = this;
            
            // Service selection - UPDATED for multiple selection
            $(document).on('click', '.kr-service-card', function() {
                const serviceId = $(this).data('service-id');
                const serviceName = $(this).data('service-name');
                const serviceDuration = parseInt($(this).data('service-duration')) || 0;
                const servicePrice = parseFloat($(this).data('service-price')) || 0;
                const serviceColor = $(this).data('service-color');
                
                // Toggle selection
                if ($(this).hasClass('selected')) {
                    // Deselect
                    $(this).removeClass('selected');
                    $(this).find('.kr-service-checkmark').hide();
                    
                    // Remove from selected services
                    self.selectedServices = self.selectedServices.filter(s => s.id !== serviceId);
                } else {
                    // Select
                    $(this).addClass('selected');
                    $(this).find('.kr-service-checkmark').show();
                    
                    // Add to selected services
                    self.selectedServices.push({
                        id: serviceId,
                        name: serviceName,
                        duration: serviceDuration,
                        price: servicePrice,
                        color: serviceColor
                    });
                }
                
                // Update selected services display
                self.updateSelectedServicesDisplay();
                
                // Enable/disable continue button
                $('#kr-continue-services').prop('disabled', self.selectedServices.length === 0);
            });
            
            // Continue from services step
            $(document).on('click', '#kr-continue-services', function() {
                if (self.selectedServices.length > 0) {
                    self.nextStep();
                }
            });
            
            // Calendar navigation
            $(document).on('click', '.kr-prev-month', function() {
                self.currentMonth.setMonth(self.currentMonth.getMonth() - 1);
                self.loadAvailableDates();
            });
            
            $(document).on('click', '.kr-next-month', function() {
                self.currentMonth.setMonth(self.currentMonth.getMonth() + 1);
                self.loadAvailableDates();
            });
            
            // Date selection - allow clicking on available, partially-booked days
            $(document).on('click', '.kr-calendar-day.available, .kr-calendar-day.partially-booked', function() {
                if ($(this).hasClass('disabled') || $(this).hasClass('fully-booked') || $(this).hasClass('closed')) {
                    return;
                }
                
                $('.kr-calendar-day').removeClass('selected');
                $(this).addClass('selected');
                self.selectedDate = $(this).data('date');
                setTimeout(() => {
                    self.loadAvailableTimes();
                    self.nextStep();
                }, 300);
            });
            
            // Time selection (only available times)
            $(document).on('click', '.kr-time-slot.kr-time-available', function() {
                if ($(this).hasClass('kr-time-booked')) {
                    return; // Don't allow clicking on booked times
                }
                
                $('.kr-time-slot').removeClass('selected');
                $(this).addClass('selected');
                self.selectedTime = $(this).data('time');
                setTimeout(() => {
                    self.updateSummary();
                    self.nextStep();
                }, 300);
            });
            
            // Remove service from selected list
            $(document).on('click', '.kr-selected-remove', function() {
                const serviceId = $(this).data('service-id');
                
                // Remove from array
                self.selectedServices = self.selectedServices.filter(s => s.id !== serviceId);
                
                // Update UI
                $(`.kr-service-card[data-service-id="${serviceId}"]`).removeClass('selected');
                $(`.kr-service-card[data-service-id="${serviceId}"] .kr-service-checkmark`).hide();
                
                // Update display
                self.updateSelectedServicesDisplay();
                
                // Enable/disable continue button
                $('#kr-continue-services').prop('disabled', self.selectedServices.length === 0);
            });
            
            // Form submission
            $('#kr-booking-form').on('submit', function(e) {
                e.preventDefault();
                self.submitBooking();
            });
            
            // Back button
            $('.kr-btn-back').on('click', function() {
                self.prevStep();
            });
            
            // New booking
            $('.kr-btn-new-booking').on('click', function() {
                self.reset();
            });
        },
        
        updateSelectedServicesDisplay: function() {
            const self = this;
            const container = $('#kr-selected-services');
            const listContainer = $('#kr-selected-list');
            
            if (self.selectedServices.length === 0) {
                container.hide();
                return;
            }
            
            container.show();
            listContainer.empty();
            
            let totalDuration = 0;
            let totalPrice = 0;
            
            self.selectedServices.forEach(function(service) {
                totalDuration += service.duration;
                totalPrice += service.price;
                
                const serviceItem = $('<div>')
                    .addClass('kr-selected-item')
                    .html(`
                        <span class="kr-selected-item-name">${service.name}</span>
                        <span class="kr-selected-item-meta">
                            <span class="kr-selected-item-duration">${service.duration} min</span>
                            <span class="kr-selected-item-price">${self.formatPrice(service.price)}</span>
                        </span>
                        <button type="button" class="kr-selected-remove" data-service-id="${service.id}" title="Odebrat">×</button>
                    `);
                
                listContainer.append(serviceItem);
            });
            
            // Update totals
            $('#kr-total-duration').text(`${totalDuration} min`);
            $('#kr-total-price').text(self.formatPrice(totalPrice));
        },
        
        formatPrice: function(price) {
            return new Intl.NumberFormat('cs-CZ', {
                style: 'currency',
                currency: 'CZK',
                minimumFractionDigits: 0,
                maximumFractionDigits: 0
            }).format(price);
        },
        
        loadServices: function() {
            const self = this;
            
            $.ajax({
                url: krAjax.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'kr_get_services',
                    nonce: krAjax.nonce
                },
                success: function(response) {
                    if (response.success) {
                        self.renderServices(response.data);
                    }
                },
                error: function() {
                    self.showError(krAjax.strings.error);
                }
            });
        },
        
        renderServices: function(services) {
            const self = this;
            const container = $('#kr-services-list');
            container.empty();
            
            services.forEach(function(service) {
                // Create color variations
                const color = service.color || '#f89820';
                
                const card = $('<div>')
                    .addClass('kr-service-card')
                    .attr('data-service-id', service.id)
                    .attr('data-service-color', color)
                    .attr('data-service-name', service.name)
                    .attr('data-service-duration', service.duration)
                    .attr('data-service-price', service.price)
                    .css({
                        '--service-color': color
                    })
                    .html(`
                        <div class="kr-service-checkmark" style="display: none;">✓</div>
                        <div class="kr-service-name">${service.name}</div>
                        <div class="kr-service-description">${service.description || ''}</div>
                        <div class="kr-service-meta">
                            <span class="kr-service-duration">${service.duration_formatted}</span>
                            <span class="kr-service-price">${service.price_formatted}</span>
                        </div>
                    `);
                
                container.append(card);
            });
        },
        
        loadAvailableDates: function() {
            const self = this;
            const fromDate = new Date(self.currentMonth.getFullYear(), self.currentMonth.getMonth(), 1);
            
            // Get total duration for all selected services
            const totalDuration = self.selectedServices.reduce((sum, service) => sum + service.duration, 0);
            
            // Add or update banner
            const serviceNames = self.selectedServices.map(s => s.name).join(', ');
            const bannerColor = self.selectedServices[0]?.color || '#f89820';
            const bannerColorDark = self.adjustColor(bannerColor, -20);
            
            if ($('.kr-calendar-banner').length === 0) {
                const banner = $('<div>')
                    .addClass('kr-calendar-banner')
                    .css({
                        '--banner-color': bannerColor,
                        '--banner-color-dark': bannerColorDark
                    })
                    .html(`
                        <div class="kr-calendar-banner-title">Vyberte termín pro: ${serviceNames}</div>
                        <div class="kr-calendar-banner-text">Celková doba: ${totalDuration} min</div>
                    `);
                
                $('.kr-calendar-container').prepend(banner);
            } else {
                $('.kr-calendar-banner').css({
                    '--banner-color': bannerColor,
                    '--banner-color-dark': bannerColorDark
                });
                $('.kr-calendar-banner-title').text(`Vyberte termín pro: ${serviceNames}`);
                $('.kr-calendar-banner-text').text(`Celková doba: ${totalDuration} min`);
            }
            
            // Get service IDs
            const serviceIds = self.selectedServices.map(s => s.id);
            
            $.ajax({
                url: krAjax.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'kr_get_available_dates',
                    nonce: krAjax.nonce,
                    service_ids: serviceIds, // Changed to array
                    total_duration: totalDuration,
                    from_date: self.formatDate(fromDate)
                },
                success: function(response) {
                    if (response.success) {
                        self.availableDates = response.data;
                        self.renderCalendar();
                    }
                }
            });
        },
        
        adjustColor: function(color, amount) {
            if (!color) return '#f89820';
            const num = parseInt(color.replace('#', ''), 16);
            const r = Math.max(0, Math.min(255, (num >> 16) + amount));
            const g = Math.max(0, Math.min(255, ((num >> 8) & 0x00FF) + amount));
            const b = Math.max(0, Math.min(255, (num & 0x0000FF) + amount));
            return '#' + ((r << 16) | (g << 8) | b).toString(16).padStart(6, '0');
        },
        
        renderCalendar: function() {
            const self = this;
            const year = self.currentMonth.getFullYear();
            const month = self.currentMonth.getMonth();
            
            // Update month display
            const monthNames = ['leden', 'únor', 'březen', 'duben', 'květen', 'červen',
                              'červenec', 'srpen', 'září', 'říjen', 'listopad', 'prosinec'];
            $('#kr-current-month').text(`${monthNames[month]} ${year}`);
            
            // Calculate calendar
            const firstDay = new Date(year, month, 1);
            const lastDay = new Date(year, month + 1, 0);
            const daysInMonth = lastDay.getDate();
            const startingDayOfWeek = firstDay.getDay();
            const adjustedStart = startingDayOfWeek === 0 ? 6 : startingDayOfWeek - 1;
            
            const container = $('#kr-calendar-days');
            container.empty();
            
            // Previous month days
            const prevMonthLastDay = new Date(year, month, 0).getDate();
            for (let i = adjustedStart; i > 0; i--) {
                container.append(`<div class="kr-calendar-day other-month">${prevMonthLastDay - i + 1}</div>`);
            }
            
            // Current month days
            const today = new Date();
            today.setHours(0, 0, 0, 0);
            
            // Get availability status for each date
            const serviceIds = self.selectedServices.map(s => s.id);
            const totalDuration = self.selectedServices.reduce((sum, service) => sum + service.duration, 0);
            
            for (let day = 1; day <= daysInMonth; day++) {
                const date = new Date(year, month, day);
                const dateStr = self.formatDate(date);
                const isPast = date < today;
                const isToday = date.getTime() === today.getTime();
                
                let classes = 'kr-calendar-day';
                
                if (isToday) {
                    classes += ' today';
                }
                
                if (isPast) {
                    classes += ' disabled';
                } else {
                    // Check availability status
                    const isAvailable = self.availableDates.includes(dateStr);
                    
                    if (isAvailable) {
                        // Fetch detailed availability to determine if fully/partially booked
                        self.checkDayAvailability(dateStr, serviceIds, totalDuration, function(status) {
                            const dayElement = $(`.kr-calendar-day[data-date="${dateStr}"]`);
                            dayElement.removeClass('available partially-booked fully-booked closed');
                            dayElement.addClass(status);
                        });
                        classes += ' available'; // Default to available, will be updated by AJAX
                    } else {
                        classes += ' closed';
                    }
                }
                
                container.append(`<div class="${classes}" data-date="${dateStr}">${day}</div>`);
            }
            
            // Next month days
            const totalCells = container.children().length;
            const remainingCells = 42 - totalCells;
            for (let day = 1; day <= remainingCells; day++) {
                container.append(`<div class="kr-calendar-day other-month">${day}</div>`);
            }
        },
        
        checkDayAvailability: function(date, serviceIds, totalDuration, callback) {
            $.ajax({
                url: krAjax.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'kr_get_available_times',
                    nonce: krAjax.nonce,
                    service_ids: serviceIds,
                    total_duration: totalDuration,
                    date: date
                },
                success: function(response) {
                    if (response.success) {
                        const data = response.data;
                        const availableCount = data.available ? data.available.length : 0;
                        const bookedCount = data.booked ? data.booked.length : 0;
                        const totalSlots = availableCount + bookedCount;
                        
                        let status = 'available';
                        if (totalSlots === 0) {
                            status = 'closed';
                        } else if (availableCount === 0) {
                            status = 'fully-booked';
                        } else if (bookedCount > 0) {
                            status = 'partially-booked';
                        }
                        
                        callback(status);
                    }
                }
            });
        },
        
        loadAvailableTimes: function() {
            const self = this;
            const serviceIds = self.selectedServices.map(s => s.id);
            const totalDuration = self.selectedServices.reduce((sum, service) => sum + service.duration, 0);
            
            $.ajax({
                url: krAjax.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'kr_get_available_times',
                    nonce: krAjax.nonce,
                    service_ids: serviceIds,
                    total_duration: totalDuration,
                    date: self.selectedDate
                },
                success: function(response) {
                    if (response.success) {
                        self.renderTimes(response.data);
                    }
                },
                error: function() {
                    self.showError(krAjax.strings.error);
                }
            });
        },
        
        renderTimes: function(timesData) {
            const container = $('#kr-times-list');
            container.empty();
            
            // timesData can be array of strings (available times) or object with available and booked
            let availableTimes = [];
            let bookedTimes = [];
            
            if (Array.isArray(timesData)) {
                availableTimes = timesData;
            } else if (timesData.available) {
                availableTimes = timesData.available;
                bookedTimes = timesData.booked || [];
            }
            
            if (availableTimes.length === 0 && bookedTimes.length === 0) {
                container.html(`
                    <div class="kr-no-times-message" style="text-align: center; padding: 40px 20px;">
                        <div style="font-size: 48px; margin-bottom: 16px;">😔</div>
                        <h4 style="color: var(--kr-text); margin-bottom: 8px;">Žádné dostupné časy</h4>
                        <p style="color: var(--kr-text-light);">Pro tento den nejsou k dispozici žádné termíny.</p>
                    </div>
                `);
                return;
            }
            
            // Show message if all times are booked
            if (availableTimes.length === 0 && bookedTimes.length > 0) {
                container.html(`
                    <div class="kr-no-times-message" style="text-align: center; padding: 40px 20px;">
                        <div style="font-size: 48px; margin-bottom: 16px;">📅</div>
                        <h4 style="color: var(--kr-text); margin-bottom: 8px;">Všechny časy jsou obsazené</h4>
                        <p style="color: var(--kr-text-light); margin-bottom: 24px;">Pro tento den jsou již všechny termíny rezervované. Zkuste prosím vybrat jiný den.</p>
                    </div>
                `);
            }
            
            // Render available times
            availableTimes.forEach(function(time) {
                const slot = $('<div>')
                    .addClass('kr-time-slot kr-time-available')
                    .attr('data-time', time)
                    .html(`<span class="kr-time-text">${time}</span><span class="kr-time-badge">Volné</span>`);
                
                container.append(slot);
            });
            
            // Render booked times (non-clickable, just for info)
            bookedTimes.forEach(function(time) {
                const slot = $('<div>')
                    .addClass('kr-time-slot kr-time-booked')
                    .attr('data-time', time)
                    .html(`<span class="kr-time-text">${time}</span><span class="kr-time-badge">Obsazeno</span>`);
                
                container.append(slot);
            });
        },
        
        updateSummary: function() {
            const self = this;
            
            // Get service details
            const serviceNames = self.selectedServices.map(s => s.name).join(', ');
            const totalDuration = self.selectedServices.reduce((sum, s) => sum + s.duration, 0);
            const totalPrice = self.selectedServices.reduce((sum, s) => sum + s.price, 0);
            
            // Format date
            const dateObj = new Date(self.selectedDate);
            const formattedDate = dateObj.toLocaleDateString('cs-CZ', {
                day: 'numeric',
                month: 'long',
                year: 'numeric'
            });
            
            // Update summary
            $('#kr-summary-services').text(serviceNames);
            $('#kr-summary-date').text(formattedDate);
            $('#kr-summary-time').text(self.selectedTime);
            $('#kr-summary-duration').text(`${totalDuration} min`);
            $('#kr-summary-price').text(self.formatPrice(totalPrice));
            
            // Update hidden fields
            const serviceIds = self.selectedServices.map(s => s.id).join(',');
            $('#kr-service-ids').val(serviceIds);
            $('#kr-booking-date').val(self.selectedDate);
            $('#kr-booking-time').val(self.selectedTime);
        },
        
        updateSuccessSummary: function() {
            const self = this;
            
            // Get service details
            const serviceNames = self.selectedServices.map(s => s.name).join(', ');
            const totalDuration = self.selectedServices.reduce((sum, s) => sum + s.duration, 0);
            const totalPrice = self.selectedServices.reduce((sum, s) => sum + s.price, 0);
            
            // Format date
            const dateObj = new Date(self.selectedDate);
            const formattedDate = dateObj.toLocaleDateString('cs-CZ', {
                day: 'numeric',
                month: 'long',
                year: 'numeric'
            });
            
            // Update success summary
            $('#kr-success-services').text(serviceNames);
            $('#kr-success-date').text(formattedDate);
            $('#kr-success-time').text(self.selectedTime);
            $('#kr-success-duration').text(`${totalDuration} min`);
            $('#kr-success-price').text(self.formatPrice(totalPrice));
        },
        
        submitBooking: function() {
            const self = this;
            const form = $('#kr-booking-form');
            const submitBtn = form.find('button[type="submit"]');
            
            // Show loading state
            const originalHtml = submitBtn.html();
            submitBtn.prop('disabled', true).html('<span class="kr-btn-icon">⏳</span> Odesílání...');
            
            $.ajax({
                url: krAjax.ajaxUrl,
                type: 'POST',
                data: form.serialize() + '&action=kr_create_booking&nonce=' + krAjax.nonce,
                success: function(response) {
                    if (response.success) {
                        // Update success summary
                        self.updateSuccessSummary();
                        
                        // Show success step
                        self.nextStep();
                        
                        // Scroll to top
                        $('html, body').animate({
                            scrollTop: $('.kr-booking-wrapper').offset().top - 50
                        }, 500);
                    } else {
                        // Show error with custom message
                        const errorMessage = response.data && response.data.errors 
                            ? response.data.errors.join('<br>') 
                            : (krAjax.strings.errorWithContact || krAjax.strings.error);
                        
                        self.showError(errorMessage);
                        submitBtn.prop('disabled', false).html(originalHtml);
                    }
                },
                error: function(xhr, status, error) {
                    // Show error with contact info
                    const errorMessage = krAjax.strings.errorWithContact || 
                        'Něco se pokazilo. Prosím kontaktujte nás telefonicky.';
                    
                    self.showError(errorMessage);
                    submitBtn.prop('disabled', false).html(originalHtml);
                }
            });
        },
        
        nextStep: function() {
            if (this.currentStep < 5) {
                $(`.kr-step[data-step="${this.currentStep}"]`).removeClass('active');
                $(`.kr-progress-step[data-step="${this.currentStep}"]`).removeClass('active');
                
                this.currentStep++;
                
                $(`.kr-step[data-step="${this.currentStep}"]`).addClass('active');
                $(`.kr-progress-step[data-step="${this.currentStep}"]`).addClass('active');
                
                // Load data for new step
                if (this.currentStep === 2) {
                    this.loadAvailableDates();
                }
            }
        },
        
        prevStep: function() {
            if (this.currentStep > 1) {
                $(`.kr-step[data-step="${this.currentStep}"]`).removeClass('active');
                $(`.kr-progress-step[data-step="${this.currentStep}"]`).removeClass('active');
                
                this.currentStep--;
                
                $(`.kr-step[data-step="${this.currentStep}"]`).addClass('active');
                $(`.kr-progress-step[data-step="${this.currentStep}"]`).addClass('active');
            }
        },
        
        reset: function() {
            this.currentStep = 1;
            this.selectedServices = [];
            this.selectedDate = null;
            this.selectedTime = null;
            
            $('.kr-step').removeClass('active');
            $('.kr-step[data-step="1"]').addClass('active');
            $('.kr-progress-step').removeClass('active');
            $('.kr-progress-step[data-step="1"]').addClass('active');
            
            $('#kr-booking-form')[0].reset();
            $('.kr-service-card, .kr-calendar-day, .kr-time-slot').removeClass('selected');
            $('.kr-service-checkmark').hide();
            $('#kr-selected-services').hide();
            $('#kr-continue-services').prop('disabled', true);
            
            // Reset submit button state
            const submitBtn = $('#kr-booking-form button[type="submit"]');
            submitBtn.prop('disabled', false).text('Potvrdit rezervaci');
            
            // Remove any error notifications
            $('.kr-error-notification').remove();
        },
        
        formatDate: function(date) {
            const year = date.getFullYear();
            const month = String(date.getMonth() + 1).padStart(2, '0');
            const day = String(date.getDate()).padStart(2, '0');
            return `${year}-${month}-${day}`;
        },
        
        showError: function(message) {
            // Create modern error notification
            const errorHtml = `
                <div class="kr-error-notification">
                    <div class="kr-error-icon">⚠️</div>
                    <div class="kr-error-content">
                        <div class="kr-error-title">Chyba při rezervaci</div>
                        <div class="kr-error-message">${message}</div>
                    </div>
                    <button class="kr-error-close" onclick="this.parentElement.remove()">×</button>
                </div>
            `;
            
            // Remove existing errors
            $('.kr-error-notification').remove();
            
            // Add error to page
            $('.kr-booking-wrapper').prepend(errorHtml);
            
            // Auto remove after 10 seconds
            setTimeout(function() {
                $('.kr-error-notification').fadeOut(300, function() {
                    $(this).remove();
                });
            }, 10000);
            
            // Scroll to error
            $('html, body').animate({
                scrollTop: $('.kr-booking-wrapper').offset().top - 100
            }, 300);
        }
    };
    
    // Initialize on document ready
    $(document).ready(function() {
        if ($('.kr-booking-wrapper').length) {
            KR_Booking.init();
        }
    });
    
})(jQuery);