<?php
/**
 * Admin services page
 */

if (!defined('ABSPATH')) {
    exit;
}

global $wpdb;
$table = $wpdb->prefix . 'kr_services';

// Handle form submissions
if (isset($_POST['kr_save_service']) && check_admin_referer('kr_service_nonce')) {
    $service_id = isset($_POST['service_id']) ? intval($_POST['service_id']) : 0;
    
    $data = array(
        'name' => sanitize_text_field($_POST['service_name']),
        'description' => sanitize_textarea_field($_POST['service_description']),
        'duration' => intval($_POST['service_duration']),
        'price' => floatval($_POST['service_price']),
        'color' => sanitize_hex_color($_POST['service_color']),
        'is_active' => isset($_POST['service_active']) ? 1 : 0
    );
    
    if ($service_id > 0) {
        // Update existing service
        $wpdb->update($table, $data, array('id' => $service_id));
        echo '<div class="notice notice-success"><p>' . __('Služba byla aktualizována.', 'kadernice-rezervace') . '</p></div>';
    } else {
        // Create new service
        $wpdb->insert($table, $data);
        echo '<div class="notice notice-success"><p>' . __('Služba byla vytvořena.', 'kadernice-rezervace') . '</p></div>';
    }
}

// Handle delete
if (isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['service_id']) && check_admin_referer('kr_delete_service_' . $_GET['service_id'])) {
    $service_id = intval($_GET['service_id']);
    $wpdb->delete($table, array('id' => $service_id));
    echo '<div class="notice notice-success"><p>' . __('Služba byla smazána.', 'kadernice-rezervace') . '</p></div>';
}

// Get service for editing
$editing_service = null;
if (isset($_GET['action']) && $_GET['action'] === 'edit' && isset($_GET['service_id'])) {
    $service_id = intval($_GET['service_id']);
    $editing_service = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE id = %d", $service_id));
}

// Get all services
$services = KR_Database::get_services(false);
?>

<div class="wrap">
    <h1 class="wp-heading-inline"><?php _e('Služby', 'kadernice-rezervace'); ?></h1>
    <a href="<?php echo admin_url('admin.php?page=kr-services&action=add'); ?>" class="page-title-action">
        <?php _e('Přidat novou', 'kadernice-rezervace'); ?>
    </a>
    <hr class="wp-header-end">
    
    <?php if (isset($_GET['action']) && ($_GET['action'] === 'add' || $_GET['action'] === 'edit')): ?>
        <!-- Add/Edit Form -->
        <div class="kr-service-form-wrapper" style="max-width: 800px; background: #fff; padding: 20px; margin: 20px 0; border: 1px solid #ccd0d4; border-radius: 4px;">
            <h2><?php echo $editing_service ? __('Upravit službu', 'kadernice-rezervace') : __('Přidat novou službu', 'kadernice-rezervace'); ?></h2>
            
            <form method="post" action="">
                <?php wp_nonce_field('kr_service_nonce'); ?>
                <input type="hidden" name="service_id" value="<?php echo $editing_service ? esc_attr($editing_service->id) : ''; ?>">
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="service_name"><?php _e('Název služby', 'kadernice-rezervace'); ?> *</label>
                        </th>
                        <td>
                            <input type="text" id="service_name" name="service_name" 
                                   value="<?php echo $editing_service ? esc_attr($editing_service->name) : ''; ?>" 
                                   class="regular-text" required>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="service_description"><?php _e('Popis', 'kadernice-rezervace'); ?></label>
                        </th>
                        <td>
                            <textarea id="service_description" name="service_description" 
                                      rows="3" class="large-text"><?php echo $editing_service ? esc_textarea($editing_service->description) : ''; ?></textarea>
                            <p class="description"><?php _e('Krátký popis služby zobrazený zákazníkům', 'kadernice-rezervace'); ?></p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="service_duration"><?php _e('Doba trvání (minuty)', 'kadernice-rezervace'); ?> *</label>
                        </th>
                        <td>
                            <input type="number" id="service_duration" name="service_duration" 
                                   value="<?php echo $editing_service ? esc_attr($editing_service->duration) : '30'; ?>" 
                                   min="15" max="480" step="15" required>
                            <p class="description"><?php _e('Jak dlouho služba trvá (v minutách)', 'kadernice-rezervace'); ?></p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="service_price"><?php _e('Cena', 'kadernice-rezervace'); ?> *</label>
                        </th>
                        <td>
                            <input type="number" id="service_price" name="service_price" 
                                   value="<?php echo $editing_service ? esc_attr($editing_service->price) : '0'; ?>" 
                                   min="0" step="0.01" required>
                            <span><?php echo esc_html(get_option('kr_currency', 'Kč')); ?></span>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="service_color"><?php _e('Barva', 'kadernice-rezervace'); ?></label>
                        </th>
                        <td>
                            <input type="color" id="service_color" name="service_color" 
                                   value="<?php echo $editing_service ? esc_attr($editing_service->color) : '#f89820'; ?>">
                            <p class="description"><?php _e('Barva pro zobrazení v kalendáři', 'kadernice-rezervace'); ?></p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row"><?php _e('Aktivní', 'kadernice-rezervace'); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="service_active" value="1" 
                                       <?php checked($editing_service ? $editing_service->is_active : 1, 1); ?>>
                                <?php _e('Služba je aktivní a zobrazuje se zákazníkům', 'kadernice-rezervace'); ?>
                            </label>
                        </td>
                    </tr>
                </table>
                
                <p class="submit">
                    <button type="submit" name="kr_save_service" class="button button-primary">
                        <?php echo $editing_service ? __('Uložit změny', 'kadernice-rezervace') : __('Přidat službu', 'kadernice-rezervace'); ?>
                    </button>
                    <a href="<?php echo admin_url('admin.php?page=kr-services'); ?>" class="button">
                        <?php _e('Zrušit', 'kadernice-rezervace'); ?>
                    </a>
                </p>
            </form>
        </div>
    <?php else: ?>
        <!-- Services List -->
        <div class="kr-services-grid">
            <?php if (empty($services)): ?>
                <p><?php _e('Zatím nemáte žádné služby.', 'kadernice-rezervace'); ?></p>
            <?php else: ?>
                <?php foreach ($services as $service): ?>
                    <div class="kr-service-item">
                        <div class="kr-service-header">
                            <h3 class="kr-service-name"><?php echo esc_html($service->name); ?></h3>
                            <span class="kr-service-status">
                                <?php if ($service->is_active): ?>
                                    <span class="dashicons dashicons-yes-alt" style="color: #22c55e;" title="<?php esc_attr_e('Aktivní', 'kadernice-rezervace'); ?>"></span>
                                <?php else: ?>
                                    <span class="dashicons dashicons-dismiss" style="color: #ef4444;" title="<?php esc_attr_e('Neaktivní', 'kadernice-rezervace'); ?>"></span>
                                <?php endif; ?>
                            </span>
                        </div>
                        
                        <?php if ($service->description): ?>
                            <p class="kr-service-description"><?php echo esc_html($service->description); ?></p>
                        <?php endif; ?>
                        
                        <div class="kr-service-meta">
                            <div class="kr-service-meta-item">
                                <span class="kr-service-meta-label"><?php _e('Doba trvání', 'kadernice-rezervace'); ?></span>
                                <span class="kr-service-meta-value"><?php echo KR_Services::format_duration($service->duration); ?></span>
                            </div>
                            <div class="kr-service-meta-item">
                                <span class="kr-service-meta-label"><?php _e('Cena', 'kadernice-rezervace'); ?></span>
                                <span class="kr-service-meta-value"><?php echo KR_Services::format_price($service->price); ?></span>
                            </div>
                            <div class="kr-service-meta-item">
                                <span class="kr-service-meta-label"><?php _e('Barva', 'kadernice-rezervace'); ?></span>
                                <span class="kr-service-meta-value">
                                    <span style="display: inline-block; width: 20px; height: 20px; background: <?php echo esc_attr($service->color); ?>; border-radius: 4px; vertical-align: middle; border: 1px solid #ddd;"></span>
                                </span>
                            </div>
                        </div>
                        
                        <div class="kr-service-actions" style="margin-top: 12px; padding-top: 12px; border-top: 1px solid #f0f0f1;">
                            <a href="<?php echo admin_url('admin.php?page=kr-services&action=edit&service_id=' . $service->id); ?>" 
                               class="button button-small">
                                <?php _e('Upravit', 'kadernice-rezervace'); ?>
                            </a>
                            <a href="<?php echo wp_nonce_url(admin_url('admin.php?page=kr-services&action=delete&service_id=' . $service->id), 'kr_delete_service_' . $service->id); ?>" 
                               class="button button-small button-link-delete"
                               onclick="return confirm('<?php esc_attr_e('Opravdu chcete smazat tuto službu?', 'kadernice-rezervace'); ?>');">
                                <?php _e('Smazat', 'kadernice-rezervace'); ?>
                            </a>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    <?php endif; ?>
</div>

<style>
.kr-services-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
    gap: 20px;
    margin-top: 20px;
}

.kr-service-item {
    background: #fff;
    border: 1px solid #ccd0d4;
    border-radius: 4px;
    padding: 20px;
}

.kr-service-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 12px;
}

.kr-service-name {
    font-size: 18px;
    font-weight: 600;
    margin: 0;
}

.kr-service-description {
    color: #646970;
    font-size: 14px;
    line-height: 1.5;
    margin: 0 0 12px;
}

.kr-service-meta {
    display: grid;
    grid-template-columns: repeat(3, 1fr);
    gap: 12px;
    margin-top: 12px;
    padding-top: 12px;
    border-top: 1px solid #f0f0f1;
}

.kr-service-meta-item {
    display: flex;
    flex-direction: column;
}

.kr-service-meta-label {
    font-size: 12px;
    color: #646970;
    margin-bottom: 4px;
}

.kr-service-meta-value {
    font-weight: 600;
    font-size: 14px;
}

@media (max-width: 782px) {
    .kr-services-grid {
        grid-template-columns: 1fr;
    }
    
    .kr-service-meta {
        grid-template-columns: 1fr;
    }
}
</style>