<?php
/**
 * Admin bookings page
 */

if (!defined('ABSPATH')) {
    exit;
}

// Get bookings
$status_filter = isset($_GET['status']) ? sanitize_text_field($_GET['status']) : '';
$date_from = isset($_GET['date_from']) ? sanitize_text_field($_GET['date_from']) : date('Y-m-d');
$date_to = isset($_GET['date_to']) ? sanitize_text_field($_GET['date_to']) : date('Y-m-d', strtotime('+30 days'));

$bookings = KR_Database::get_bookings(array(
    'date_from' => $date_from,
    'date_to' => $date_to,
    'status' => $status_filter,
    'limit' => 100
));

$statuses = array(
    'pending' => __('Čekající', 'kadernice-rezervace'),
    'confirmed' => __('Potvrzené', 'kadernice-rezervace'),
    'cancelled' => __('Zrušené', 'kadernice-rezervace'),
    'completed' => __('Dokončené', 'kadernice-rezervace')
);
?>

<div class="wrap">
    <h1 class="wp-heading-inline"><?php _e('Rezervace', 'kadernice-rezervace'); ?></h1>
    <hr class="wp-header-end">
    
    <!-- Filters -->
    <div class="kr-admin-filters">
        <form method="get" action="">
            <input type="hidden" name="page" value="kr-bookings">
            
            <label for="date-from"><?php _e('Od:', 'kadernice-rezervace'); ?></label>
            <input type="date" id="date-from" name="date_from" value="<?php echo esc_attr($date_from); ?>">
            
            <label for="date-to"><?php _e('Do:', 'kadernice-rezervace'); ?></label>
            <input type="date" id="date-to" name="date_to" value="<?php echo esc_attr($date_to); ?>">
            
            <label for="status"><?php _e('Status:', 'kadernice-rezervace'); ?></label>
            <select id="status" name="status">
                <option value=""><?php _e('Všechny', 'kadernice-rezervace'); ?></option>
                <?php foreach ($statuses as $key => $label): ?>
                    <option value="<?php echo esc_attr($key); ?>" <?php selected($status_filter, $key); ?>>
                        <?php echo esc_html($label); ?>
                    </option>
                <?php endforeach; ?>
            </select>
            
            <button type="submit" class="button"><?php _e('Filtrovat', 'kadernice-rezervace'); ?></button>
        </form>
    </div>
    
    <!-- Bookings Table -->
    <table class="wp-list-table widefat fixed striped">
        <thead>
            <tr>
                <th><?php _e('ID', 'kadernice-rezervace'); ?></th>
                <th><?php _e('Zákazník', 'kadernice-rezervace'); ?></th>
                <th><?php _e('Služba', 'kadernice-rezervace'); ?></th>
                <th><?php _e('Datum', 'kadernice-rezervace'); ?></th>
                <th><?php _e('Čas', 'kadernice-rezervace'); ?></th>
                <th><?php _e('Status', 'kadernice-rezervace'); ?></th>
                <th><?php _e('Akce', 'kadernice-rezervace'); ?></th>
            </tr>
        </thead>
        <tbody>
            <?php if (empty($bookings)): ?>
                <tr>
                    <td colspan="7" style="text-align: center;">
                        <?php _e('Žádné rezervace nenalezeny', 'kadernice-rezervace'); ?>
                    </td>
                </tr>
            <?php else: ?>
                <?php foreach ($bookings as $booking): ?>
                    <tr>
                        <td><?php echo esc_html($booking->id); ?></td>
                        <td>
                            <strong><?php echo esc_html($booking->customer_name); ?></strong><br>
                            <small>
                                <?php echo esc_html($booking->customer_email); ?><br>
                                <?php echo esc_html($booking->customer_phone); ?>
                            </small>
                        </td>
                        <td><?php echo esc_html($booking->service_name); ?></td>
                        <td><?php echo date_i18n('d.m.Y', strtotime($booking->booking_date)); ?></td>
                        <td><?php echo date_i18n('H:i', strtotime($booking->booking_time)); ?></td>
                        <td>
                            <select class="kr-status-select" data-booking-id="<?php echo esc_attr($booking->id); ?>">
                                <?php foreach ($statuses as $key => $label): ?>
                                    <option value="<?php echo esc_attr($key); ?>" <?php selected($booking->status, $key); ?>>
                                        <?php echo esc_html($label); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                        <td>
                            <button type="button" class="button button-small kr-view-booking" data-booking-id="<?php echo esc_attr($booking->id); ?>">
                                📋 <?php _e('Detail', 'kadernice-rezervace'); ?>
                            </button>
                            <button type="button" class="button button-small kr-send-email" 
                                    data-booking-id="<?php echo esc_attr($booking->id); ?>"
                                    data-customer-email="<?php echo esc_attr($booking->customer_email); ?>">
                                ✉️ <?php _e('Email', 'kadernice-rezervace'); ?>
                            </button>
                            <button type="button" class="button button-small button-link-delete kr-delete-booking" data-booking-id="<?php echo esc_attr($booking->id); ?>">
                                🗑️ <?php _e('Smazat', 'kadernice-rezervace'); ?>
                            </button>
                        </td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
    </table>
</div>

<!-- Booking Detail Modal -->
<div id="kr-booking-modal" class="kr-modal" style="display: none;">
    <div class="kr-modal-overlay"></div>
    <div class="kr-modal-content">
        <div class="kr-modal-header">
            <h2><?php _e('Detail rezervace', 'kadernice-rezervace'); ?></h2>
            <button type="button" class="kr-modal-close">&times;</button>
        </div>
        <div class="kr-modal-body" id="kr-booking-detail">
            <p><?php _e('Načítání...', 'kadernice-rezervace'); ?></p>
        </div>
    </div>
</div>

<!-- Email Modal -->
<div id="kr-email-modal" class="kr-modal" style="display: none;">
    <div class="kr-modal-overlay"></div>
    <div class="kr-modal-content">
        <div class="kr-modal-header">
            <h2><?php _e('Odeslat email zákazníkovi', 'kadernice-rezervace'); ?></h2>
            <button type="button" class="kr-modal-close">&times;</button>
        </div>
        <div class="kr-modal-body">
            <form id="kr-email-form">
                <input type="hidden" id="email-booking-id" name="booking_id">
                <table class="form-table">
                    <tr>
                        <th><label for="email-to"><?php _e('Příjemce:', 'kadernice-rezervace'); ?></label></th>
                        <td><input type="email" id="email-to" name="email_to" class="regular-text" readonly></td>
                    </tr>
                    <tr>
                        <th><label for="email-subject"><?php _e('Předmět:', 'kadernice-rezervace'); ?></label></th>
                        <td><input type="text" id="email-subject" name="email_subject" class="regular-text" 
                                   value="<?php echo esc_attr(get_option('kr_business_name')); ?> - Informace o rezervaci"></td>
                    </tr>
                    <tr>
                        <th><label for="email-message"><?php _e('Zpráva:', 'kadernice-rezervace'); ?></label></th>
                        <td><textarea id="email-message" name="email_message" rows="8" class="large-text"></textarea></td>
                    </tr>
                </table>
                <p class="submit">
                    <button type="submit" class="button button-primary"><?php _e('Odeslat email', 'kadernice-rezervace'); ?></button>
                    <button type="button" class="button kr-modal-close"><?php _e('Zrušit', 'kadernice-rezervace'); ?></button>
                </p>
            </form>
        </div>
    </div>
</div>

<style>
.kr-admin-filters {
    background: #fff;
    padding: 15px;
    margin: 20px 0;
    border: 1px solid #ccd0d4;
    box-shadow: 0 1px 1px rgba(0,0,0,.04);
}

.kr-admin-filters form {
    display: flex;
    gap: 10px;
    align-items: center;
    flex-wrap: wrap;
}

.kr-admin-filters label {
    font-weight: 600;
}

.kr-admin-filters input[type="date"],
.kr-admin-filters select {
    padding: 4px 8px;
}

.kr-modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    z-index: 100000;
}

.kr-modal-overlay {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.7);
}

.kr-modal-content {
    position: relative;
    background: #fff;
    max-width: 600px;
    margin: 50px auto;
    border-radius: 4px;
    box-shadow: 0 5px 15px rgba(0,0,0,0.3);
    max-height: 90vh;
    overflow-y: auto;
}

.kr-modal-header {
    padding: 20px;
    border-bottom: 1px solid #ddd;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.kr-modal-header h2 {
    margin: 0;
    font-size: 20px;
}

.kr-modal-close {
    background: none;
    border: none;
    font-size: 28px;
    cursor: pointer;
    color: #666;
    padding: 0;
    width: 30px;
    height: 30px;
    line-height: 1;
}

.kr-modal-close:hover {
    color: #000;
}

.kr-modal-body {
    padding: 20px;
}

.kr-booking-info {
    background: #f9f9f9;
    padding: 15px;
    border-radius: 4px;
    margin-bottom: 15px;
}

.kr-booking-info-row {
    display: flex;
    padding: 8px 0;
    border-bottom: 1px solid #e5e5e5;
}

.kr-booking-info-row:last-child {
    border-bottom: none;
}

.kr-booking-info-label {
    font-weight: 600;
    width: 150px;
    color: #666;
}

.kr-booking-info-value {
    flex: 1;
    color: #000;
}
</style>

<script>
jQuery(document).ready(function($) {
    // Update booking status
    $('.kr-status-select').on('change', function() {
        const bookingId = $(this).data('booking-id');
        const newStatus = $(this).val();
        
        if (!confirm('<?php _e('Opravdu chcete změnit status této rezervace?', 'kadernice-rezervace'); ?>')) {
            return;
        }
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'kr_update_booking_status',
                nonce: '<?php echo wp_create_nonce('kr-admin-nonce'); ?>',
                booking_id: bookingId,
                status: newStatus
            },
            success: function(response) {
                if (response.success) {
                    alert('<?php _e('Status byl aktualizován', 'kadernice-rezervace'); ?>');
                } else {
                    alert(response.data.message || '<?php _e('Chyba při aktualizaci', 'kadernice-rezervace'); ?>');
                }
            },
            error: function() {
                alert('<?php _e('Chyba při komunikaci se serverem', 'kadernice-rezervace'); ?>');
            }
        });
    });
    
    // View booking detail
    $('.kr-view-booking').on('click', function() {
        const bookingId = $(this).data('booking-id');
        
        $('#kr-booking-detail').html('<p><?php _e('Načítání...', 'kadernice-rezervace'); ?></p>');
        $('#kr-booking-modal').fadeIn(200);
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'kr_get_booking_detail',
                nonce: '<?php echo wp_create_nonce('kr-admin-nonce'); ?>',
                booking_id: bookingId
            },
            success: function(response) {
                if (response.success) {
                    $('#kr-booking-detail').html(response.data.html);
                } else {
                    $('#kr-booking-detail').html('<p><?php _e('Chyba při načítání detailu', 'kadernice-rezervace'); ?></p>');
                }
            },
            error: function() {
                $('#kr-booking-detail').html('<p><?php _e('Chyba při komunikaci se serverem', 'kadernice-rezervace'); ?></p>');
            }
        });
    });
    
    // Send email
    $('.kr-send-email').on('click', function() {
        const bookingId = $(this).data('booking-id');
        const customerEmail = $(this).data('customer-email');
        
        $('#email-booking-id').val(bookingId);
        $('#email-to').val(customerEmail);
        $('#kr-email-modal').fadeIn(200);
    });
    
    // Submit email form
    $('#kr-email-form').on('submit', function(e) {
        e.preventDefault();
        
        const submitBtn = $(this).find('button[type="submit"]');
        submitBtn.prop('disabled', true).text('<?php _e('Odesílání...', 'kadernice-rezervace'); ?>');
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: $(this).serialize() + '&action=kr_send_customer_email&nonce=<?php echo wp_create_nonce('kr-admin-nonce'); ?>',
            success: function(response) {
                if (response.success) {
                    alert('<?php _e('Email byl odeslán', 'kadernice-rezervace'); ?>');
                    $('#kr-email-modal').fadeOut(200);
                    $('#kr-email-form')[0].reset();
                } else {
                    alert(response.data.message || '<?php _e('Chyba při odesílání emailu', 'kadernice-rezervace'); ?>');
                }
                submitBtn.prop('disabled', false).text('<?php _e('Odeslat email', 'kadernice-rezervace'); ?>');
            },
            error: function() {
                alert('<?php _e('Chyba při komunikaci se serverem', 'kadernice-rezervace'); ?>');
                submitBtn.prop('disabled', false).text('<?php _e('Odeslat email', 'kadernice-rezervace'); ?>');
            }
        });
    });
    
    // Delete booking
    $('.kr-delete-booking').on('click', function() {
        if (!confirm('<?php _e('Opravdu chcete smazat tuto rezervaci? Tato akce je nevratná.', 'kadernice-rezervace'); ?>')) {
            return;
        }
        
        const bookingId = $(this).data('booking-id');
        const row = $(this).closest('tr');
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'kr_delete_booking',
                nonce: '<?php echo wp_create_nonce('kr-admin-nonce'); ?>',
                booking_id: bookingId
            },
            success: function(response) {
                if (response.success) {
                    row.fadeOut(300, function() {
                        $(this).remove();
                    });
                } else {
                    alert(response.data.message || '<?php _e('Chyba při mazání', 'kadernice-rezervace'); ?>');
                }
            },
            error: function() {
                alert('<?php _e('Chyba při komunikaci se serverem', 'kadernice-rezervace'); ?>');
            }
        });
    });
    
    // Close modals
    $('.kr-modal-close, .kr-modal-overlay').on('click', function() {
        $('.kr-modal').fadeOut(200);
    });
});
</script>