<?php
/**
 * Email notifications  DIRECT SMTP (NO wp_mail)
 */

if (!defined('ABSPATH')) {
    exit;
}

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

class KR_Email {

    /* =========================================================
     * SMTP  PHPMailer (OBCHZ WORDPRESS)
     * ========================================================= */
    private static function get_mailer(): ?PHPMailer {

        if (!class_exists(PHPMailer::class)) {
            require_once ABSPATH . WPINC . '/PHPMailer/PHPMailer.php';
            require_once ABSPATH . WPINC . '/PHPMailer/SMTP.php';
            require_once ABSPATH . WPINC . '/PHPMailer/Exception.php';
        }

        $host   = get_option('kr_smtp_host');
        $user   = get_option('kr_smtp_username');
        $pass   = get_option('kr_smtp_password');
        $port   = (int) get_option('kr_smtp_port', 587);
        $secure = get_option('kr_smtp_encryption', 'tls');

        if (!$host || !$user || !$pass) {
            error_log('? KR_Email: chyb SMTP daje');
            return null;
        }

        try {
            $mail = new PHPMailer(true);
            $mail->isSMTP();
            $mail->Host       = $host;
            $mail->SMTPAuth   = true;
            $mail->Username   = $user;
            $mail->Password   = $pass;
            $mail->Port       = $port;
            $mail->SMTPSecure = $secure;
            $mail->CharSet    = 'UTF-8';

            $from_email = get_option('kr_smtp_from_email', $user);
            $from_name  = get_option('kr_smtp_from_name', get_bloginfo('name'));

            $mail->setFrom($from_email, $from_name);
            $mail->isHTML(true);

            return $mail;

        } catch (Exception $e) {
            error_log('? KR_Email SMTP init chyba: ' . $e->getMessage());
            return null;
        }
    }

    private static function send_mail(string $to, string $subject, string $html): bool {

        $mail = self::get_mailer();
        if (!$mail) {
            return false;
        }

        if (empty($html)) {
            error_log("? KR_Email: tlo zprvy je przdn pro $to | $subject");
            $html = '<p>(dn zprva nebyla vygenerovna)</p>';
        }

        try {
            $mail->clearAddresses();
            $mail->addAddress($to);
            $mail->Subject = $subject;
            $mail->Body    = $html;
            $mail->send();

            error_log("? KR_Email odeslno: $to | $subject");
            return true;

        } catch (Exception $e) {
            error_log('? KR_Email send chyba: ' . $e->getMessage());
            return false;
        }
    }
    /**
   Test SMTP connection  pmo pes PHPMailer
 */
public static function test_smtp_connection(): array {
    // kam se m poslat testovac e-mail
    $to = get_option('kr_business_email');
    if (!$to) {
        $to = get_option('kr_smtp_from_email', get_option('kr_smtp_username'));
    }

    $subject = __('Test SMTP - Kadenice Rezervace', 'kadernice-rezervace');
    $message = '<p>' . __('Toto je testovac email. Pokud jste ho obdreli, SMTP funguje sprvn!', 'kadernice-rezervace') . '</p>';

    // pouijeme vlastn metodu send_mail()
    $ok = self::send_mail($to, $subject, $message);

    if ($ok) {
        return ['success' => true, 'message' => 'Email byl spn odesln.'];
    } else {
        return ['success' => false, 'message' => 'Nepodailo se odeslat email pes SMTP.'];
    }
}


   /**
 * Send booking confirmation to customer  pes PHPMailer + nvratov hodnota
 */
public static function send_booking_confirmation($booking_id): bool {
    if (!get_option('kr_customer_email_on_booking', 1)) {
        return false;
    }

    $booking = KR_Database::get_booking($booking_id);
    if (!$booking) {
        return false;
    }

    $to      = $booking->customer_email;
    $subject = sprintf(__('Potvrzen rezervace - %s', 'kadernice-rezervace'), get_option('kr_business_name'));
    $message = self::get_booking_confirmation_template($booking);

    // poslme pmo pes PHPMailer
    $ok = self::send_mail($to, $subject, $message);

    // pokud se m poslat i adminovi
    if ($ok && get_option('kr_admin_email_on_booking', 1)) {
        self::send_admin_notification($booking);
    }

    // SMS notifikace
    if ($ok) {
        KR_SMS::send_booking_confirmation($booking_id);
        KR_SMS::send_admin_notification($booking_id);
    }

    return $ok;
}
    /**
 * Send notification to admin  pes PHPMailer + nvratov hodnota
 */
private static function send_admin_notification($booking): bool {
    // kam se m poslat admin notifikace
    $to = get_option('kr_business_email');
    if (!$to) {
        $to = get_option('kr_smtp_from_email', get_option('kr_smtp_username'));
    }

    $subject = sprintf(__('Nov rezervace - %s', 'kadernice-rezervace'), $booking->customer_name ?? '');
    $message = self::get_admin_notification_template($booking);

    // poslme pmo pes PHPMailer
    $ok = self::send_mail($to, $subject, $message);

    return $ok;
}
   /**
 * Send multi-service booking confirmation to customer  pes PHPMailer + nvratov hodnota
 */
public static function send_multi_service_confirmation($main_booking_id, $service_names, $total_duration, $total_price): bool {
    if (!get_option('kr_customer_email_on_booking', 1)) {
        return false;
    }

    $booking = KR_Database::get_booking($main_booking_id);
    if (!$booking) {
        return false;
    }

    $to      = $booking->customer_email;
    $subject = sprintf(__('Potvrzen rezervace - %s', 'kadernice-rezervace'), get_option('kr_business_name'));
    $message = self::get_multi_service_confirmation_template($booking, $service_names, $total_duration, $total_price);

    // poslme pmo pes PHPMailer
    $ok = self::send_mail($to, $subject, $message);

    // pokud se m poslat i adminovi
    if ($ok && get_option('kr_admin_email_on_booking', 1)) {
        self::send_multi_service_admin_notification($booking, $service_names, $total_duration, $total_price);
    }

    return $ok;
}
    
    /**
 * Send multi-service notification to admin  pes PHPMailer + nvratov hodnota
 */
private static function send_multi_service_admin_notification($booking, $service_names, $total_duration, $total_price): bool {
    // kam se m poslat admin notifikace
    $to = get_option('kr_business_email');
    if (!$to) {
        $to = get_option('kr_smtp_from_email', get_option('kr_smtp_username'));
    }

    $subject = sprintf(__('Nov rezervace (vce slueb) - %s', 'kadernice-rezervace'), $booking->customer_name ?? '');
    $message = self::get_multi_service_confirmation_template($booking, $service_names, $total_duration, $total_price);

    // poslme pmo pes PHPMailer
    $ok = self::send_mail($to, $subject, $message);

    return $ok;
}
    
    /**
     * Get multi-service confirmation email template
     */
    private static function get_multi_service_confirmation_template($booking, $service_names, $total_duration, $total_price) {
        $date = date_i18n(get_option('kr_date_format', 'd.m.Y'), strtotime($booking->booking_date));
        $time = date_i18n(get_option('kr_time_format', 'H:i'), strtotime($booking->booking_time));
        $currency = get_option('kr_currency', 'K');
        
        ob_start();
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: linear-gradient(135deg, #f89820, #e08718); color: white; padding: 30px; text-align: center; border-radius: 8px 8px 0 0; }
                .content { background: #f9f9f9; padding: 30px; }
                .booking-details { background: white; padding: 20px; border-radius: 8px; margin: 20px 0; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
                .detail-row { padding: 12px 0; border-bottom: 1px solid #eee; display: flex; justify-content: space-between; }
                .detail-row:last-child { border-bottom: none; }
                .detail-label { font-weight: bold; color: #666; }
                .detail-value { color: #333; }
                .services-list { background: #fff3e0; padding: 15px; border-radius: 8px; margin: 15px 0; }
                .service-item { padding: 8px 0; color: #333; }
                .service-item:before { content: '? '; color: #f89820; font-weight: bold; }
                .total-section { background: #e3f2fd; padding: 15px; border-radius: 8px; margin: 15px 0; }
                .total-row { display: flex; justify-content: space-between; padding: 8px 0; font-size: 16px; }
                .total-row strong { color: #1976d2; }
                .footer { text-align: center; padding: 20px; color: #666; font-size: 14px; }
                .highlight { color: #f89820; font-weight: bold; }
            </style>
        </head>
        <body>
            <div class="container">
                <div class="header">
                    <h1><?php echo esc_html(get_option('kr_business_name')); ?></h1>
                    <p style="margin: 10px 0 0 0; font-size: 18px;">Potvrzen rezervace</p>
                </div>
                <div class="content">
                    <p><?php printf(__('Dobr den %s,', 'kadernice-rezervace'), '<strong>' . esc_html($booking->customer_name) . '</strong>'); ?></p>
                    <p><?php _e('Dkujeme za vai rezervaci. Ne naleznete detaily:', 'kadernice-rezervace'); ?></p>
                    
                    <div class="booking-details">
                        <div class="detail-row">
                            <span class="detail-label"><?php _e('Datum:', 'kadernice-rezervace'); ?></span>
                            <span class="detail-value"><?php echo esc_html($date); ?></span>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label"><?php _e('as:', 'kadernice-rezervace'); ?></span>
                            <span class="detail-value"><?php echo esc_html($time); ?></span>
                        </div>
                    </div>
                    
                    <div class="services-list">
                        <h3 style="margin-top: 0; color: #f89820;"><?php _e('Vybran sluby:', 'kadernice-rezervace'); ?></h3>
                        <?php foreach ($service_names as $service_name): ?>
                            <div class="service-item"><?php echo esc_html($service_name); ?></div>
                        <?php endforeach; ?>
                    </div>
                    
                    <div class="total-section">
                        <div class="total-row">
                            <span><?php _e('Celkov doba:', 'kadernice-rezervace'); ?></span>
                            <strong><?php echo esc_html($total_duration); ?> min</strong>
                        </div>
                        <div class="total-row">
                            <span><?php _e('Celkov cena:', 'kadernice-rezervace'); ?></span>
                            <strong><?php echo number_format($total_price, 0, ',', ' '); ?> <?php echo esc_html($currency); ?></strong>
                        </div>
                    </div>
                    
                    <?php if ($booking->notes): ?>
                    <div class="booking-details">
                        <div class="detail-row">
                            <span class="detail-label"><?php _e('Poznmka:', 'kadernice-rezervace'); ?></span>
                        </div>
                        <p style="margin: 10px 0 0 0;"><?php echo nl2br(esc_html($booking->notes)); ?></p>
                    </div>
                    <?php endif; ?>
                    
                    <p><?php _e('Tme se na vai nvtvu!', 'kadernice-rezervace'); ?></p>
                    
                    <?php if (get_option('kr_business_phone')): ?>
                    <p>
                        <?php _e('V ppad dotaz ns kontaktujte na:', 'kadernice-rezervace'); ?><br>
                        <strong><?php echo esc_html(get_option('kr_business_phone')); ?></strong>
                    </p>
                    <?php endif; ?>
                </div>
                <div class="footer">
                    <p><?php echo esc_html(get_option('kr_business_name')); ?></p>
                    <?php if (get_option('kr_business_email')): ?>
                    <p><?php echo esc_html(get_option('kr_business_email')); ?></p>
                    <?php endif; ?>
                </div>
            </div>
        </body>
        </html>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Get customer confirmation email template
     */
    private static function get_booking_confirmation_template($booking) {
        $date = date_i18n(get_option('kr_date_format', 'd.m.Y'), strtotime($booking->booking_date));
        $time = date_i18n(get_option('kr_time_format', 'H:i'), strtotime($booking->booking_time));
        
        ob_start();
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: #f89820; color: white; padding: 20px; text-align: center; }
                .content { background: #f9f9f9; padding: 30px; }
                .booking-details { background: white; padding: 20px; border-radius: 8px; margin: 20px 0; }
                .detail-row { padding: 10px 0; border-bottom: 1px solid #eee; }
                .detail-label { font-weight: bold; color: #666; }
                .footer { text-align: center; padding: 20px; color: #666; font-size: 14px; }
            </style>
        </head>
        <body>
            <div class="container">
                <div class="header">
                    <h1><?php echo esc_html(get_option('kr_business_name')); ?></h1>
                </div>
                <div class="content">
                    <h2><?php _e('Potvrzen rezervace', 'kadernice-rezervace'); ?></h2>
                    <p><?php printf(__('Dobr den %s,', 'kadernice-rezervace'), esc_html($booking->customer_name)); ?></p>
                    <p><?php _e('Dkujeme za vai rezervaci. Ne naleznete detaily:', 'kadernice-rezervace'); ?></p>
                    
                    <div class="booking-details">
                        <div class="detail-row">
                            <span class="detail-label"><?php _e('Sluba:', 'kadernice-rezervace'); ?></span>
                            <?php echo esc_html($booking->service_name); ?>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label"><?php _e('Datum:', 'kadernice-rezervace'); ?></span>
                            <?php echo esc_html($date); ?>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label"><?php _e('as:', 'kadernice-rezervace'); ?></span>
                            <?php echo esc_html($time); ?>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label"><?php _e('Doba trvn:', 'kadernice-rezervace'); ?></span>
                            <?php echo KR_Services::format_duration($booking->duration); ?>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label"><?php _e('Cena:', 'kadernice-rezervace'); ?></span>
                            <?php echo KR_Services::format_price($booking->price); ?>
                        </div>
                    </div>
                    
                    <?php if ($booking->notes): ?>
                    <p><strong><?php _e('Vae poznmka:', 'kadernice-rezervace'); ?></strong><br>
                    <?php echo nl2br(esc_html($booking->notes)); ?></p>
                    <?php endif; ?>
                    
                    <p><?php _e('Tme se na vai nvtvu!', 'kadernice-rezervace'); ?></p>
                </div>
                <div class="footer">
                    <p><?php echo esc_html(get_option('kr_business_name')); ?></p>
                    <?php if (get_option('kr_business_phone')): ?>
                    <p><?php _e('Tel:', 'kadernice-rezervace'); ?> <?php echo esc_html(get_option('kr_business_phone')); ?></p>
                    <?php endif; ?>
                    <p><?php echo esc_html(get_option('kr_business_email')); ?></p>
                </div>
            </div>
        </body>
        </html>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Get admin notification template
     */
    private static function get_admin_notification_template($booking) {
        $date = date_i18n(get_option('kr_date_format', 'd.m.Y'), strtotime($booking->booking_date));
        $time = date_i18n(get_option('kr_time_format', 'H:i'), strtotime($booking->booking_time));
        
        ob_start();
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: #2d4a45; color: white; padding: 20px; }
                .content { background: #f9f9f9; padding: 30px; }
                .booking-details { background: white; padding: 20px; border-radius: 8px; }
                .detail-row { padding: 10px 0; border-bottom: 1px solid #eee; }
                .detail-label { font-weight: bold; color: #666; }
            </style>
        </head>
        <body>
            <div class="container">
                <div class="header">
                    <h2><?php _e('Nov rezervace', 'kadernice-rezervace'); ?></h2>
                </div>
                <div class="content">
                    <div class="booking-details">
                        <div class="detail-row">
                            <span class="detail-label"><?php _e('Zkaznk:', 'kadernice-rezervace'); ?></span>
                            <?php echo esc_html($booking->customer_name); ?>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label"><?php _e('Email:', 'kadernice-rezervace'); ?></span>
                            <?php echo esc_html($booking->customer_email); ?>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label"><?php _e('Telefon:', 'kadernice-rezervace'); ?></span>
                            <?php echo esc_html($booking->customer_phone); ?>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label"><?php _e('Sluba:', 'kadernice-rezervace'); ?></span>
                            <?php echo esc_html($booking->service_name); ?>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label"><?php _e('Datum:', 'kadernice-rezervace'); ?></span>
                            <?php echo esc_html($date); ?>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label"><?php _e('as:', 'kadernice-rezervace'); ?></span>
                            <?php echo esc_html($time); ?>
                        </div>
                        <?php if ($booking->notes): ?>
                        <div class="detail-row">
                            <span class="detail-label"><?php _e('Poznmka:', 'kadernice-rezervace'); ?></span><br>
                            <?php echo nl2br(esc_html($booking->notes)); ?>
                        </div>
                        <?php endif; ?>
                    </div>
                    <p style="margin-top: 20px;">
                        <a href="<?php echo admin_url('admin.php?page=kadernice-rezervace&booking_id=' . $booking->id); ?>" 
                           style="background: #f89820; color: white; padding: 10px 20px; text-decoration: none; border-radius: 4px; display: inline-block;">
                            <?php _e('Zobrazit rezervaci', 'kadernice-rezervace'); ?>
                        </a>
                    </p>
                </div>
            </div>
        </body>
        </html>
        <?php
        return ob_get_clean();
    }
}