<?php
/**
 * Plugin Name: RezervacePro.CZ
 * Description: Moderní český rezervační systém pro WordPress s vlastním kalendářem, administrací a notifikacemi.
 * Version: 1.0
 * Author: Jitka Andersová
 * Text Domain: moje-rezervace
 */

if (!defined('ABSPATH')) exit;

// ---------------------------------
// Načtení textdomain pro překlady
// ---------------------------------
add_action('plugins_loaded', function() {
    load_plugin_textdomain('moje-rezervace', false, dirname(plugin_basename(__FILE__)) . '/languages/');
});

// ---------------------------------
// Načtení licence jako první
// ---------------------------------
require_once plugin_dir_path(__FILE__) . 'includes/license-handler.php';

// 2️⃣ Zobraz hlášku o licenci v adminu
add_action('admin_notices', function() {
    if (!bs_is_license_active()) {
        echo '<div class="notice notice-warning is-dismissible">
            <p><strong>Plugin RezervacePro.cz</strong> vyžaduje licenční klíč. 
            <a href="' . esc_url(admin_url('admin.php?page=bs_license')) . '">Zadejte jej zde</a>.</p>
        </div>';
    }
});
// ---------------------------------
// Načtení funkcí a handlerů
// ---------------------------------
require_once plugin_dir_path(__FILE__) . 'includes/functions.php';
require_once plugin_dir_path(__FILE__) . 'includes/ajax-handlers.php';
require_once plugin_dir_path(__FILE__) . 'includes/email-handler.php';
require_once plugin_dir_path(__FILE__) . 'includes/license-handler.php';

// ---------------------------------
// Shortcode pro kalendář s kontrolou licence
// ---------------------------------
add_shortcode('nova_rezervace_kalendar', function($atts) {
    if (!function_exists('bs_is_license_active')) return '';
    if (!bs_is_license_active()) {
        return '<div class="bs-license-error" style="color:red; font-weight:bold;">' . __('Licence není aktivní. Prosím zadejte licenční klíč v administraci.', 'moje-rezervace') . '</div>';
    }
    if (function_exists('nova_rezervace_calendar_shortcode')) {
        return nova_rezervace_calendar_shortcode($atts);
    }
    return '';
});

// ---------------------------------
// Frontend styly a skripty
// ---------------------------------
function bs_enqueue_frontend_assets() {
    if (!function_exists('bs_is_license_active') || !bs_is_license_active()) return;

    wp_enqueue_style('bs-style', plugin_dir_url(__FILE__) . 'assets/style.css', [], '1.0');
    wp_enqueue_script('bs-calendar-js', plugin_dir_url(__FILE__) . 'assets/calendar.js', ['jquery'], '1.0', true);
    wp_enqueue_script('bs-formular-js', plugin_dir_url(__FILE__) . 'assets/formulare.js', ['jquery'], '1.0', true);
    wp_enqueue_script('bs-notifications', plugin_dir_url(__FILE__) . 'assets/notifications.js', [], '1.0', true);

    wp_localize_script('bs-calendar-js', 'bs_ajax', [
        'url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('mr_nonce')
    ]);
}
wp_localize_script('bs-calendar-js', 'mrNastaveniKalendaru', array(
    'banner' => get_option('mr_calendar_banner', ''),
    'barvy' => array(
        'volny' => get_option('mr_calendar_color', '#3a87ad'),
    ),
    'font' => get_option('mr_calendar_font', 'Arial, sans-serif'),
    'zobrazLegend' => get_option('mr_calendar_show_legend', 1),
    'zobrazWeekend' => get_option('mr_calendar_show_weekend', 1),
));
add_action('wp_enqueue_scripts', 'bs_enqueue_frontend_assets');

function bs_enqueue_admin_assets() {
    wp_enqueue_style('bs-admin-style', plugin_dir_url(__FILE__) . 'assets/admin-style.css', [], '1.0');
    wp_enqueue_style('mr-admin-mobile-style', plugin_dir_url(__FILE__) . 'assets/mobil.css', [], '1.0');

    wp_enqueue_script('bs-tabs-js', plugin_dir_url(__FILE__) . 'assets/tabs.js', [], '1.0', true);
    wp_enqueue_script('bs-notifications', plugin_dir_url(__FILE__) . 'assets/notifications.js', [], '1.0', true);
    wp_enqueue_script('bs-mobile-tooltip-js', plugin_dir_url(__FILE__) . 'assets/mobil-tooltip.js', ['jquery'], '1.0', true);
    wp_enqueue_script('bs-nastaveni-kalendar-js', plugin_dir_url(__FILE__) . 'templates/modules/nastaveni-kalendaru.js', ['jquery'], '1.0', true );

    // ✅ Tady správně
    if (function_exists('wp_enqueue_media')) {
        wp_enqueue_media();
    }
}
add_action('admin_enqueue_scripts', 'bs_enqueue_admin_assets');
// ---------------------------------
// Administrace pluginu a menu
// ---------------------------------
function bs_admin_menu() {
    
   
   add_action('load-bs_admin', function() {
    // Pokud licence není aktivní, přesměruj na stránku licence
    if (!bs_is_license_active()) {
        wp_redirect(admin_url('admin.php?page=bs_license'));
    } else {
        wp_redirect(admin_url('admin.php?page=bs_rezervace'));
    }
    exit;
});
   
   // Hlavní menu – bez volání funkce, jen ikona a název
     add_menu_page(
        __('Rezervační systém', 'nova-rezervace'),
        __('Rezervace', 'nova-rezervace'),
        'manage_options',
        'bs_admin',
        '__return_null', // žádná funkce, jen placeholder
        'dashicons-calendar-alt',
        25
    );

    // Podmenu – jednotlivé stránky
    add_submenu_page(
        'bs_admin',
        __('Rezervace', 'nova-rezervace'),
        __('Rezervace', 'nova-rezervace'),
        'manage_options',
        'bs_rezervace',
        function() {
            include plugin_dir_path(__FILE__) . 'admin/admin-times.php';
        }
    );

    add_submenu_page(
        'bs_admin',
        __('Hromadné rezervace', 'moje-rezervace'),
        __('Hromadné rezervace', 'moje-rezervace'),
        'manage_options',
        'bs_hromadne',
        function() {
            include plugin_dir_path(__FILE__) . 'admin/admin-hromadne.php';
        }
    );

    add_submenu_page(
        'bs_admin',
        __('Seznam rezervací', 'moje-rezervace'),
        __('Seznam rezervací', 'moje-rezervace'),
        'manage_options',
        'bs_seznam',
        function() {
            include plugin_dir_path(__FILE__) . 'admin/reservations-list.php';
        }
    );

    add_submenu_page(
        'bs_admin',
        __('Nastavení pošty', 'moje-rezervace'),
        __('Nastavení pošty', 'moje-rezervace'),
        'manage_options',
        'bs_email',
        function() {
            include plugin_dir_path(__FILE__) . 'admin/admin-email-settings.php';
        }
    );

    // Podmenu – Licenční klíč
    add_submenu_page(
    'bs_admin',
    __('Licenční klíč', 'moje-rezervace'),
    __('Licenční klíč', 'moje-rezervace'),
    'manage_options',
    'bs_license',
    'bs_license_page'
);


    add_submenu_page(
        'bs_admin',
        __('Nastavení kalendáře', 'moje-rezervace'),
        __('Nastavení kalendáře', 'moje-rezervace'),
        'manage_options',
        'bs_calendar',
        function() {
            include plugin_dir_path(__FILE__) . 'admin/admin-calendar-settings.php';
        }
    );
}
add_action('admin_menu', 'bs_admin_menu');

// Přesměrování hlavní položky na první podmenu
add_action('load-bs_admin', function() {
    wp_redirect(admin_url('admin.php?page=bs_rezervace'));
    exit;
});

// Odstranění duplicitního podmenu
add_action('admin_menu', function() {
    global $submenu;
    if (isset($submenu['bs_admin'][0])) unset($submenu['bs_admin'][0]);
}, 999);

// ---------------------------------
// SMTP nastavení emailů
// ---------------------------------
add_action('phpmailer_init', function($phpmailer) {
    $smtp_host   = get_option('smtp_host', '');
    $smtp_port   = get_option('smtp_port', 587);
    $smtp_user   = get_option('smtp_user', '');
    $smtp_pass   = get_option('smtp_pass', '');
    $smtp_secure = get_option('smtp_secure', 'tls');
    $from_email  = get_option('from_email', $smtp_user);
    $from_name   = get_option('from_name', 'Rezervace');

    if ($smtp_host && $smtp_user && $smtp_pass) {
        $phpmailer->isSMTP();
        $phpmailer->Host       = $smtp_host;
        $phpmailer->SMTPAuth   = true;
        $phpmailer->Port       = $smtp_port;
        $phpmailer->Username   = $smtp_user;
        $phpmailer->Password   = $smtp_pass;
        $phpmailer->SMTPSecure = ($smtp_secure !== 'none') ? $smtp_secure : '';
        $phpmailer->setFrom($from_email, $from_name);
    }
});



// ---------------------------------
// AJAX handler pro vlastní email
// ---------------------------------
add_action('wp_ajax_mr_send_email', 'mr_send_email_handler');
function mr_send_email_handler() {
    check_admin_referer('bs_custom_email','bs_nonce');

    $to      = sanitize_email($_POST['custom_email_to'] ?? '');
    $subject = sanitize_text_field($_POST['custom_email_subject'] ?? '');
    $message = wp_kses_post($_POST['custom_email_message'] ?? '');
    $headers = ['Content-Type: text/html; charset=UTF-8'];

    if(wp_mail($to, $subject, $message, $headers)) {
        wp_send_json_success();
    } else {
        wp_send_json_error(__('Nepodařilo se odeslat e-mail.', 'nova-rezervace'));
    }
}

// ---------------------------------
// Aktivace pluginu – vytvoření tabulky a uvítací hláška
// ---------------------------------
register_activation_hook(__FILE__, 'bs_create_rezervace_table');
register_activation_hook(__FILE__, function() {
    add_option('bs_show_shortcode_notice', 1);
});

add_action('admin_notices', 'bs_shortcode_info_box');
function bs_shortcode_info_box() {
    if (!get_option('bs_show_shortcode_notice')) return;
    delete_option('bs_show_shortcode_notice');

    $create_page_url = admin_url('post-new.php?post_type=page&content=' . urlencode('[nova_rezervace_kalendar]'));
    ?>
    <div class="notice notice-success is-dismissible">
        <p><strong><?php echo esc_html__('Nova Rezervace – plugin byl úspěšně aktivován 🎉', 'nova-rezervace'); ?></strong></p>
        <p>
            <?php echo esc_html__('Pro zobrazení rezervačního kalendáře vložte do stránky nebo příspěvku tento shortcode:', 'nova-rezervace'); ?><br>
            <code id="bs-shortcode">[nova_rezervace_kalendar]</code>
            <button class="button button-primary" onclick="bsCopyShortcode()"><?php echo esc_html__('Zkopírovat shortcode', 'nova-rezervace'); ?></button>
        </p>
        <p>
            👉 <?php echo esc_html__('Doporučujeme vložit ho do nové stránky s názvem například', 'nova-rezervace'); ?> <em>Rezervace</em>.
        </p>
        <p>
            <a href="<?php echo esc_url($create_page_url); ?>" class="button button-secondary">
                ➕ <?php echo esc_html__('Vytvořit novou stránku s kalendářem', 'nova-rezervace'); ?>
            </a>
        </p>
        <div id="bs-copy-success" style="display:none; margin-top:10px; color:green; font-weight:bold;">
            ✅ <?php echo esc_html__('Shortcode byl úspěšně zkopírován!', 'nova-rezervace'); ?>
        </div>
    </div>
    <script>
        function bsCopyShortcode() {
            const shortcode = document.getElementById("bs-shortcode").innerText;
            navigator.clipboard.writeText(shortcode).then(() => {
                const msg = document.getElementById("bs-copy-success");
                msg.style.display = "block";
                setTimeout(() => { msg.style.display = "none"; }, 2000);
            }).catch(err => { console.error("Nepodařilo se zkopírovat shortcode", err); });
        }
    </script>
    <?php
}

function mr_load_textdomain() {
    load_plugin_textdomain(
        'moje-rezervace',
        false,
        dirname(plugin_basename(__FILE__)) . '/languages'
    );
}
add_action('plugins_loaded', 'mr_load_textdomain');
