<?php
if (!defined('ABSPATH')) exit;

/**
 * Soubor: includes/license-handler.php
 * Účel: Správa demo licence pluginu a propojení s tvou administrací
 */

// -----------------------------------------------------
// 🔹 1. Kontrola licence / demo
// -----------------------------------------------------
function bs_is_license_active() {
    $license_key = get_option('bs_license_key', '');
    $demo_expire = get_option('bs_demo_expire', '');
    if (!empty($demo_expire)) {
        return strtotime($demo_expire) >= time();
    }
    return !empty($license_key);
}

function bs_is_full_license() {
    $license_key = get_option('bs_license_key', '');
    return !empty($license_key) && strpos($license_key, 'DEMO-') !== 0;
}

// -----------------------------------------------------
// 🔹 2. Automatické nastavení demo
// -----------------------------------------------------
function bs_setup_demo_if_needed() {
    $license_key = get_option('bs_license_key', '');
    $demo_expire = get_option('bs_demo_expire', '');
    $demo_start  = get_option('bs_demo_start', '');

    if (empty($license_key) && empty($demo_expire)) {
        $demo_key = 'DEMO-' . strtoupper(wp_generate_password(10, false, false));
        $start_date = current_time('Y-m-d');
        $expire_date = date('Y-m-d', strtotime('+14 days'));

        update_option('bs_license_key', $demo_key);
        update_option('bs_demo_start', $start_date);
        update_option('bs_demo_expire', $expire_date);

        // Odešli info do tvé admin tabulky
        bs_send_demo_info($demo_key);
    }
}
add_action('admin_init', 'bs_setup_demo_if_needed');

// -----------------------------------------------------
// 🔹 3. PHPMailer – SMTP konfigurace
// -----------------------------------------------------
add_action('phpmailer_init', function($phpmailer) {
    $smtp_host   = get_option('smtp_host', '');
    $smtp_port   = get_option('smtp_port', 587);
    $smtp_user   = get_option('smtp_user', '');
    $smtp_pass   = get_option('smtp_pass', '');
    $smtp_secure = get_option('smtp_secure', 'tls');
    $from_email  = get_option('from_email', $smtp_user);
    $from_name   = get_option('from_name', 'RezervacePro.CZ');

    if ($smtp_host && $smtp_user && $smtp_pass) {
        $phpmailer->isSMTP();
        $phpmailer->Host       = $smtp_host;
        $phpmailer->SMTPAuth   = true;
        $phpmailer->Port       = $smtp_port;
        $phpmailer->Username   = $smtp_user;
        $phpmailer->Password   = $smtp_pass;
        $phpmailer->SMTPSecure = ($smtp_secure !== 'none') ? $smtp_secure : '';
        $phpmailer->setFrom($from_email, $from_name);
    }
});

// -----------------------------------------------------
// 🔹 4. Odešle informace o stažení demo pluginu
// -----------------------------------------------------
function bs_send_demo_info($demo_key) {
    $remote_url = 'https://www.test.rezervace-atelier.eu/wp-admin/admin-post.php?action=bs_demo_register';
    
    $data = [
        'name'        => sanitize_text_field(get_option('bs_demo_user_name', 'Neznámý')),
        'email'       => sanitize_email(get_option('bs_demo_user_email', 'demo@example.com')),
        'product_name'=> 'Rezervační systém',
        'license_key' => $demo_key,
        'status'      => 'demo staženo',
        'created_at'  => current_time('mysql'),
        'nonce'       => wp_create_nonce('bs_demo_register')
    ];

    wp_remote_post($remote_url, [
        'method'    => 'POST',
        'timeout'   => 10,
        'body'      => $data
    ]);
}

// -----------------------------------------------------
// 🔹 5. Stránka pro demo a aktivaci licence
// -----------------------------------------------------
function bs_license_page() {
    if (!current_user_can('manage_options')) wp_die(__('Nemáte oprávnění', 'rezervacepro'));

    $current_key = get_option('bs_license_key', '');
    $demo_expire = get_option('bs_demo_expire', '');
    $demo_start  = get_option('bs_demo_start', '');
    $is_full     = bs_is_full_license();
    ?>
    <div class="wrap">
        <h1>Licence – RezervacePro.CZ</h1>

        <p><strong>Demo verze:</strong> 14 dní zdarma k testování.</p>

        <?php if ($is_full): ?>
            <p style="color:#46b450;">✅ Plná licence aktivní</p>
        <?php elseif ($demo_expire && $demo_start): ?>
            <p>🧪 Demo je aktivní od <?php echo esc_html($demo_start); ?> do <?php echo esc_html($demo_expire); ?>.</p>
        <?php endif; ?>

        <?php if (!$is_full): ?>
            <p><a href="https://www.test.rezervace-atelier.eu/plna-verze-koupit/" class="button button-primary" target="_blank">🛒 Koupit plnou verzi</a></p>
        <?php endif; ?>

        <!-- 🔹 Formulář pro aktivaci licence -->
        <form method="post" action="<?php echo admin_url('admin-post.php'); ?>">
            <?php wp_nonce_field('bs_activate_license', 'bs_activate_license_nonce'); ?>
            <p>
                <label for="bs_license_key">Aktivační kód:</label><br>
                <input type="text" id="bs_license_key" name="bs_license_key" value="<?php echo esc_attr($current_key); ?>" style="width:300px;" required>
            </p>
            <p>
                <button type="submit" class="button button-primary">Aktivovat a uložit</button>
            </p>
            <input type="hidden" name="action" value="bs_activate_license">
        </form>

        <?php if (isset($_GET['license']) && $_GET['license'] === 'activated'): ?>
            <div class="notice notice-success is-dismissible">
                <p>✅ Plugin vám byl aktivován.</p>
            </div>
        <?php endif; ?>
    </div>
    <?php
}

// -----------------------------------------------------
// 🔹 6. Přidání stránky do menu
// -----------------------------------------------------
add_action('admin_menu', function() {
    add_menu_page(
        'Licence RezervacePro.CZ',
        'Licence RezervacePro.CZ',
        'manage_options',
        'bs-license',
        'bs_license_page',
        'dashicons-lock',
        99
    );
});

// -----------------------------------------------------
// 🔹 7. Handler pro aktivaci licence
// -----------------------------------------------------
add_action('admin_post_bs_activate_license', 'bs_handle_activate_license');

function bs_handle_activate_license() {
    if (!isset($_POST['bs_activate_license_nonce']) || !wp_verify_nonce($_POST['bs_activate_license_nonce'], 'bs_activate_license')) {
        wp_die('Neplatný požadavek.');
    }

    $license_key = sanitize_text_field($_POST['bs_license_key']);

    // Uložení licence do options
    update_option('bs_license_key', $license_key);

    // Přesměrování zpět s hláškou
    wp_redirect(add_query_arg('license', 'activated', admin_url('admin.php?page=bs-license')));
    exit;
}

// -----------------------------------------------------
// 🔹 8. Upozornění o demo režimu
// -----------------------------------------------------
add_action('admin_notices', function() {
    $demo_start  = get_option('bs_demo_start', '');
    $demo_expire = get_option('bs_demo_expire', '');

    if (!empty($demo_start) && !empty($demo_expire) && strtotime($demo_expire) >= time()) {
        echo '<div class="notice notice-info" style="border-left-color:#0073aa;">
            🧪 Demo je aktivní od ' . esc_html($demo_start) . ' do ' . esc_html($demo_expire) . '.
        </div>';
    } elseif (!empty($demo_expire) && strtotime($demo_expire) < time()) {
        echo '<div class="notice notice-error" style="border-left-color:#dc3232;">
            ⏰ Platnost demo vypršela. Pro další používání si zakupte plnou licenci.
        </div>';
    }
});
