<?php
if (!defined('ABSPATH')) exit;

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

/**
 * Inicializace PHPMaileru napřímo (NE wp_mail)
 */
function mr_get_mailer(): ?PHPMailer {

    if (!class_exists(PHPMailer::class)) {
        require_once ABSPATH . WPINC . '/PHPMailer/PHPMailer.php';
        require_once ABSPATH . WPINC . '/PHPMailer/SMTP.php';
        require_once ABSPATH . WPINC . '/PHPMailer/Exception.php';
    }

    $host = get_option('smtp_host');
    $user = get_option('smtp_user');
    $pass = get_option('smtp_pass');
    $port = (int) get_option('smtp_port', 587);
    $secure = get_option('smtp_secure', 'tls');

    if (!$host || !$user || !$pass) {
        error_log('❌ SMTP: chybí údaje');
        return null;
    }

    $mail = new PHPMailer(true);

    try {
        $mail->isSMTP();
        $mail->Host       = $host;
        $mail->SMTPAuth   = true;
        $mail->Username   = $user;
        $mail->Password   = $pass;
        $mail->Port       = $port;
        $mail->SMTPSecure = $secure;
        $mail->CharSet    = 'UTF-8';

        $mail->setFrom($user, get_bloginfo('name'));
        $mail->isHTML(true);

        return $mail;

    } catch (Exception $e) {
        error_log('❌ PHPMailer init chyba: ' . $e->getMessage());
        return null;
    }
}

/**
 * Odeslání e-mailu
 */
function mr_posli_email($to, $subject, $html): bool {

    $mail = mr_get_mailer();
    if (!$mail) return false;

    try {
        $mail->clearAddresses();
        $mail->addAddress($to);
        $mail->Subject = $subject;
        $mail->Body    = $html;

        $mail->send();
        error_log("✅ Email odeslán: $to | $subject");
        return true;

    } catch (Exception $e) {
        error_log("❌ Email CHYBA: " . $e->getMessage());
        return false;
    }
}

// ------------------------------------------------------
// Načtení šablony e-mailu
// ------------------------------------------------------
function mr_nacti_admin_sablonu($soubor, $data = []) {
    $cesta = plugin_dir_path(__FILE__) . 'email/' . $soubor;
    if (!file_exists($cesta)) {
        error_log("❌ Šablona nenalezena (admin): $cesta");
        return '';
    }
    $obsah = file_get_contents($cesta);
    foreach ($data as $k => $v) {
        $obsah = str_replace('{{'.$k.'}}', $v ?? '', $obsah);
    }
    return $obsah;
}

function mr_nacti_client_sablonu($soubor, $data = []) {
    $cesta = plugin_dir_path(__FILE__) . 'email-frontend/' . $soubor;
    if (!file_exists($cesta)) {
        error_log("❌ Šablona nenalezena (front-end): $cesta");
        return '';
    }
    $obsah = file_get_contents($cesta);
    foreach ($data as $k => $v) {
        $obsah = str_replace('{{'.$k.'}}', $v ?? '', $obsah);
    }
    return $obsah;
}

// ------------------------------------------------------
// Front-end: potvrzení klientovi
// ------------------------------------------------------
function mr_posli_potvrzeni($reservation){
    $body = mr_nacti_client_sablonu('rezervacni-formular.html', [
        'jmeno'    => $reservation->jmeno ?? '',
        'datum'    => $reservation->datum ?? '',
        'cas'      => $reservation->cas ?? '',
        'mobil'    => $reservation->mobil ?? '',
        'zprava'   => $reservation->zprava ?? ''
    ]);
    return mr_posli_email($reservation->email ?? '', 'Potvrzení rezervace', $body);
}

// Front-end: notifikace administrátorovi
function mr_posli_admin_notifikaci($reservation){
    $body_admin = mr_nacti_client_sablonu('rezervace.html', [
        'datum'  => $reservation->datum ?? '',
        'cas'    => $reservation->cas ?? '',
        'jmeno'  => $reservation->jmeno ?? '',
        'email'  => $reservation->email ?? '',
        'mobil'  => $reservation->mobil ?? '',
        'zprava' => $reservation->zprava ?? ''
    ]);
    return mr_posli_email(get_option('from_email', 'admin@example.com'), 'Nová rezervace', $body_admin);
}

// ------------------------------------------------------
// Admin: AJAX odeslání e-mailu z administrace
// ------------------------------------------------------
add_action('wp_ajax_mr_send_email', function() {
    if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');

    check_ajax_referer('bs_nonce');

    $to      = sanitize_email($_POST['custom_email_to'] ?? '');
    $subject = sanitize_text_field($_POST['custom_email_subject'] ?? '');
    $body    = wp_kses_post($_POST['custom_email_message'] ?? '');

    if (!$to || !$subject || !$body) wp_send_json_error(['message' => 'Chybí potřebná data']);

    $sent = mr_posli_email($to, $subject, $body);

    if ($sent) wp_send_json_success(['message' => 'E-mail byl úspěšně odeslán']);
    else wp_send_json_error(['message' => 'Nepodařilo se odeslat e-mail']);
});

// ------------------------------------------------------
// Admin: AJAX načtení šablony pro náhled
// ------------------------------------------------------
add_action('wp_ajax_mr_load_email_template', function() {
    if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');

    check_ajax_referer('bs_nonce');

    $template    = sanitize_text_field($_POST['template'] ?? '');
    $reservation = json_decode(stripslashes($_POST['reservation'] ?? '{}'), true);

    $content = mr_nacti_admin_sablonu($template, $reservation);

    if (!$content) wp_send_json_error('Šablona nenalezena nebo prázdná');

    wp_send_json_success(['body' => $content]);
});

// ------------------------------------------------------
// Připomínky a zrušení rezervace (front-end)
// ------------------------------------------------------
function mr_posli_pripominku($reservation){
    $body = mr_nacti_client_sablonu('pripominka.html', [
        'jmeno' => $reservation->jmeno ?? '',
        'datum' => $reservation->datum ?? '',
        'cas'   => $reservation->cas ?? ''
    ]);
    return mr_posli_email($reservation->email ?? '', 'Připomínka rezervace', $body);
}

function mr_posli_ruseni($reservation){
    $body = mr_nacti_client_sablonu('ruseni.html', [
        'jmeno' => $reservation->jmeno ?? '',
        'datum' => $reservation->datum ?? '',
        'cas'   => $reservation->cas ?? ''
    ]);
    mr_posli_email($reservation->email ?? '', 'Zrušení rezervace', $body);
    mr_posli_email(get_option('from_email', 'admin@example.com'), 'Rezervace byla zrušena', $body);
}
