<?php
if (!defined('ABSPATH')) exit;

global $wpdb;
$table_name = $wpdb->prefix . 'rezervace';
$reservations = $wpdb->get_results("SELECT * FROM $table_name ORDER BY datum ASC, cas ASC");

// --- E-mailové šablony ---
$email_templates = [
    'rezervacni-formular.html' => __('Potvrzení rezervace klientovi', 'nova-rezervace'),
    'rezervace.html'           => __('Nová rezervace adminovi', 'nova-rezervace'),
    'pripominka.html'          => __('Připomínka rezervace', 'nova-rezervace'),
    'ruseni.html'              => __('Zrušení rezervace', 'nova-rezervace')
];
?>

<div class="wrap">

    <!-- Filtr + Export (jeden select pro obojí) -->
    <div style="display:flex; align-items:center; gap:15px; margin-bottom:15px;">
        <label for="reservation-filter"><?php _e('Zobrazit:', 'nova-rezervace'); ?></label>
        <select id="reservation-filter" name="stav">
            <option value="vse"><?php _e('Vše', 'nova-rezervace'); ?></option>
            <option value="volny"><?php _e('Volné', 'nova-rezervace'); ?></option>
            <option value="obsazeno"><?php _e('Obsazené', 'nova-rezervace'); ?></option>
        </select>

        <button type="button" id="export-btn" class="button button-secondary">
            <?php _e('Exportovat seznam', 'nova-rezervace'); ?>
        </button>
    </div>

    <!-- Seznam rezervací -->
    <h2><?php _e('Seznam rezervací', 'nova-rezervace'); ?></h2>

    <?php if ($reservations): ?>
        
        <table class="widefat fixed striped" id="reservations-table">
            <thead>
                <tr>
                    <th>ID</th><th>Jméno</th><th>Email</th><th>Mobil</th>
                    <th>Datum</th><th>Čas</th><th>Poznámka</th><th>Stav</th><th>Akce</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($reservations as $r): ?>
                    <tr data-dostupnost="<?php echo esc_attr($r->dostupnost); ?>" id="row-<?php echo esc_attr($r->id); ?>">
                        <td><?php echo esc_html($r->id); ?></td>
                        <td><?php echo esc_html($r->jmeno); ?></td>
                        <td><?php echo esc_html($r->email); ?></td>
                        <td><?php echo esc_html($r->mobil); ?></td>
                        <td><?php echo esc_html($r->datum); ?></td>
                        <td><?php echo esc_html($r->cas); ?></td>
                        <td><?php echo esc_html($r->zprava); ?></td>
                        <td><?php echo esc_html($r->dostupnost); ?></td>
                        <td>
                            <!-- Ponecháme stejná data-nonce a data-id jako dosud -->
                            <button class="button button-small btn-delete"
                                    data-id="<?php echo esc_attr($r->id); ?>"
                                    data-nonce="<?php echo wp_create_nonce('bs_delete_reservation'); ?>">
                                <?php _e('Vymazat', 'nova-rezervace'); ?>
                            </button>

                            <?php if (!empty($r->email)): ?>
                                <button type="button" class="button button-small btn-email"
                                        data-reservation='<?php echo json_encode($r); ?>'>
                                    <?php _e('Odeslat e-mail', 'nova-rezervace'); ?>
                                </button>
                            <?php endif; ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php else: ?>
        <p><?php _e('Žádné rezervace zatím nejsou.', 'nova-rezervace'); ?></p>
    <?php endif; ?>
</div>

<!-- Modal pro odeslání e-mailu (zachováno tak, jak jsi měla) -->

<div id="customEmailModal" style="display:none; position:fixed; top:10%; left:50%; transform:translateX(-50%);
     background:#fff; border:1px solid #ccc; padding:20px; z-index:10000; width:600px; box-shadow:0 0 10px rgba(0,0,0,0.3);">
    <h3><?php _e('Odeslat e-mail', 'nova-rezervace'); ?></h3>
    <form id="emailForm">
        <?php wp_nonce_field('bs_nonce','bs_nonce'); ?>
        <input type="hidden" name="custom_email_id" id="custom_email_id">

        <p>
            <label for="custom_email_type"><?php _e('Typ e-mailu:', 'nova-rezervace'); ?></label><br>
            <select name="custom_email_type" id="custom_email_type" style="width:100%;">
                <?php foreach ($email_templates as $file=>$label): ?>
                    <option value="<?php echo esc_attr($file); ?>"><?php echo esc_html($label); ?></option>
                <?php endforeach; ?>
            </select>
        </p>

        <p>
            <label for="custom_email_to"><?php _e('Příjemce:', 'nova-rezervace'); ?></label><br>
            <input type="email" name="custom_email_to" id="custom_email_to" style="width:100%;" required>
        </p>

        <p>
            <label for="custom_email_subject"><?php _e('Předmět:', 'nova-rezervace'); ?></label><br>
            <input type="text" name="custom_email_subject" id="custom_email_subject" style="width:100%;" required>
        </p>

        <p>
            <label for="custom_email_message_display"><?php _e('Zpráva:', 'nova-rezervace'); ?></label><br>
            <div id="custom_email_message_display" contenteditable="true"
                 style="border:1px solid #ccc; padding:10px; min-height:150px; background:#f9f9f9;"></div>
            <input type="hidden" name="custom_email_message" id="custom_email_message">
        </p>

        <p style="text-align:right;">
            <button type="submit" class="button button-primary"><?php _e('Odeslat','nova-rezervace'); ?></button>
            <button type="button" class="button" onclick="closeEmailModal();"><?php _e('Zavřít','nova-rezervace'); ?></button>
        </p>
    </form>
</div>

<script>
document.addEventListener('DOMContentLoaded', function(){

    // bezpečně získat nonce (pokud existuje jako input v DOMu)
    const bs_nonce_input = document.querySelector('input[name="bs_nonce"]') || document.getElementById('bs_nonce');
    const bs_nonce = bs_nonce_input ? bs_nonce_input.value : '';

    // --- Dynamický filtr (client-side) ---
    const filterSelect = document.getElementById('reservation-filter');
    if (filterSelect) {
        filterSelect.addEventListener('change', function(){
            const value = this.value;
            document.querySelectorAll('#reservations-table tbody tr').forEach(row=>{
                const stav = (row.dataset.dostupnost || '').trim();
                row.style.display = (value === 'vse' || stav === value) ? '' : 'none';
            });
        });
    }

    // --- Export (client-side: vytvoří CSV z aktuálně zobrazených řádků) ---
    const exportBtn = document.getElementById('export-btn');
    if (exportBtn) {
        exportBtn.addEventListener('click', function(){
            try {
                const rows = Array.from(document.querySelectorAll('#reservations-table tbody tr'));
                // vezmeme jen viditelné řádky
                const visibleRows = rows.filter(r => r.offsetParent !== null);
                const csvLines = [];
                // hlavička
                csvLines.push(['ID','Jméno','Email','Mobil','Datum','Čas','Poznámka','Stav'].join(','));

                visibleRows.forEach(row => {
                    const cells = row.querySelectorAll('td');
                    const values = [];
                    for (let i = 0; i < 8; i++) { // první 8 buněk (bez akcí)
                        const cell = cells[i];
                        let text = cell ? cell.innerText.trim() : '';
                        text = text.replace(/(\r\n|\n|\r)/gm, ' '); // odstranit nové řádky
                        text = text.replace(/"/g, '""'); // escape "
                        values.push('"' + text + '"');
                    }
                    csvLines.push(values.join(','));
                });

                const csvContent = csvLines.join('\r\n');
                const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
                const now = new Date().toISOString().slice(0,10);
                const filename = 'rezervace-' + now + '.csv';
                const link = document.createElement('a');
                link.href = URL.createObjectURL(blob);
                link.download = filename;
                document.body.appendChild(link);
                link.click();
                link.remove();
            } catch (e) {
                console.error('Chyba při exportu CSV:', e);
                alert('Chyba při exportu CSV. Podívejte se do konzole pro detaily.');
            }
        });
    }

    // --- E-mail modal & templates (ponechá stávající chování) ---
    const modal = document.getElementById('customEmailModal');
    const emailForm = document.getElementById('emailForm');
    const messageBox = document.getElementById('custom_email_message_display');
    const hiddenMessage = document.getElementById('custom_email_message');

    function openEmailModal(reservation){
        if(!reservation) return;
        document.getElementById('custom_email_id').value = reservation.id;
        document.getElementById('custom_email_to').value = reservation.email || '';
        document.getElementById('custom_email_subject').value = 'Rezervace pro ' + (reservation.jmeno || '');
        modal.dataset.reservation = JSON.stringify(reservation);
        modal.style.display = 'block';
        // načíst šablonu
        loadTemplate(document.getElementById('custom_email_type').value, reservation);
    }

    function closeEmailModal(){ modal.style.display = 'none'; }
    window.closeEmailModal = closeEmailModal;

    function loadTemplate(template, reservation){
        const data = new URLSearchParams();
        data.append('action','mr_load_email_template');
        data.append('template', template);
        data.append('reservation', JSON.stringify(reservation));
        if (bs_nonce) data.append('_wpnonce', bs_nonce);

        fetch(ajaxurl, { method:'POST', body: data })
        .then(res => res.json())
        .then(res => {
            if(res.success){
                messageBox.innerHTML = res.data.body || '';
            } else {
                alert('Chyba při načítání šablony: ' + (res.data || res));
            }
        })
        .catch(err => {
            console.error('Chyba mr_load_email_template:', err);
            alert('Chyba při načítání šablony (viz konzole).');
        });
    }

    // napojit tlačítka "Odeslat e-mail"
    document.querySelectorAll('.btn-email').forEach(btn=>{
        btn.addEventListener('click', function(){
            const data = this.dataset.reservation || '{}';
            try {
                openEmailModal(JSON.parse(data));
            } catch(e){
                console.error('Chyba parsing reservation data:', e, data);
            }
        });
    });

    // změna typu šablony
    const typeSelect = document.getElementById('custom_email_type');
    if (typeSelect) {
        typeSelect.addEventListener('change', function(){
            const reservation = modal.dataset.reservation ? JSON.parse(modal.dataset.reservation) : {};
            loadTemplate(this.value, reservation);
        });
    }

    // odeslání e-mailu přes AJAX (ponechat stejnou akci)
    if (emailForm) {
        emailForm.addEventListener('submit', function(e){
            e.preventDefault();
            hiddenMessage.value = messageBox.innerHTML || '';

            const data = new FormData(emailForm);
            data.append('action','mr_send_email');
            if (bs_nonce) data.append('_wpnonce', bs_nonce);

            fetch(ajaxurl, { method:'POST', body: data })
            .then(res => res.json())
            .then(res => {
                if (res.success) {
                    alert(res.data.message || 'E-mail byl odeslán');
                    closeEmailModal();
                } else {
                    alert(res.data?.message || 'Chyba při odesílání e-mailu');
                }
            })
            .catch(err => {
                console.error('Chyba mr_send_email:', err);
                alert('Chyba při odesílání e-mailu (viz konzole).');
            });
        });
    }

    // --- Mazání rezervace (AJAX) - ponechá existující endpoint "bs_delete_reservation" ---
    document.querySelectorAll('.btn-delete').forEach(btn=>{
        btn.addEventListener('click', function(){
            if(!confirm('Opravdu smazat rezervaci?')) return;
            const id = this.dataset.id;
            const nonce = this.dataset.nonce;

            const data = new URLSearchParams();
            data.append('action','bs_delete_reservation');
            data.append('id', id);
            if (nonce) data.append('_wpnonce', nonce);

            fetch(ajaxurl, { method:'POST', body: data })
            .then(res => res.json())
            .then(res => {
                if (res.success) {
                    const row = document.getElementById('row-' + id);
                    if (row) row.remove();
                } else {
                    alert('Chyba při mazání: ' + (res.data?.message || res.data || 'Neznámá chyba'));
                }
            })
            .catch(err => {
                console.error('Chyba bs_delete_reservation:', err);
                alert('Chyba při mazání rezervace (viz konzole).');
            });
        });
    });

}); // DOMContentLoaded
</script>
